-----------------------------------------------------------------------
--
--  File:        demo.adb
--  Description: Win95/NT console support demonstration
--  Rev:         0.2
--  Date:        08-june-1999
--  Author:      Jerry van Dijk
--  Mail:        jdijk@acm.org
--
--  Copyright (c) Jerry van Dijk, 1997, 1998, 1999
--  Billie Holidaystraat 28
--  2324 LK  LEIDEN
--  THE NETHERLANDS
--  tel int + 31 71 531 43 65
--
--  Permission granted to use for any purpose, provided this copyright
--  remains attached and unmodified.
--
--  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
--  IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
--
-----------------------------------------------------------------------

--          ***********************************************
--          ** WARNING: ALGORITHM DELIBERATELY CRIPPLED! **
--          ***********************************************

WITH NT_Console;
USE NT_Console;
WITH Ada.Text_IO;
USE Ada.Text_IO;
WITH Ada.Strings.Fixed;
USE Ada.Strings.Fixed;
WITH Ada.Characters.Handling;
USE Ada.Characters.Handling;

PROCEDURE Demo IS

   TYPE Card_Company IS
         (Unknown,
          Mastercard,
          Visa,
          Amex,
          Diners);

   TYPE Credit_Card_Type IS
      RECORD
         Card          : Card_Company;
         Prefix_String : String (1 .. 3);
         Prefix_Length : Positive RANGE 1 .. 3;
      END RECORD;

   Max_Length : CONSTANT := 20;

   -- Do *NOT* change order of table
   Card_Table : CONSTANT
   ARRAY (1 .. 16) OF Credit_Card_Type :=
      (Credit_Card_Type'(Diners,     "300", 3),
      Credit_Card_Type'(Diners,     "301", 3),
      Credit_Card_Type'(Diners,     "302", 3),
      Credit_Card_Type'(Diners,     "303", 3),
      Credit_Card_Type'(Diners,     "304", 3),
      Credit_Card_Type'(Diners,     "305", 3),
      Credit_Card_Type'(Amex,       "34 ", 2),
      Credit_Card_Type'(Diners,     "36 ", 2),
      Credit_Card_Type'(Amex,       "37 ", 2),
      Credit_Card_Type'(Diners,     "38 ", 2),
      Credit_Card_Type'(Mastercard, "51 ", 2),
      Credit_Card_Type'(Mastercard, "52 ", 2),
      Credit_Card_Type'(Mastercard, "53 ", 2),
      Credit_Card_Type'(Mastercard, "54 ", 2),
      Credit_Card_Type'(Mastercard, "55 ", 2),
      Credit_Card_Type'(Visa,       "4  ", 1)
      );

   Length  : Natural;
   Company : Card_Company;
   Again   : Boolean                  := True;
   Number  : String (1 .. Max_Length);

   PROCEDURE Display_Banner IS
   BEGIN
      Clear_Screen (Blue);
      Set_Foreground (Blue);
      Set_Background (Yellow);
      Goto_XY (27, 0);
      Put ("* CREDIT CARD VALIDATION *");
      Set_Background (Blue);
      Set_Foreground (Light_Red);
      Goto_XY (20, 2);
      Put ("WARNING: for demonstration purposes only!");
      Set_Foreground (Yellow);
      Goto_XY (12, 8);
      Put ("Please enter card number below, press <ENTER> when ready");
   END Display_Banner;

   FUNCTION Ask_Again RETURN Boolean IS
      Result : Boolean;
      Key    : Character := 'X';
   BEGIN
      Set_Foreground (White);
      Set_Background (Blue);
      Goto_XY (31, 20);
      Put ("Run again (Y/N) ?");
      WHILE Key /= 'Y' AND Key /= 'N' LOOP
         Key := To_Upper (Get_Key);
      END LOOP;
      IF Key = 'Y' THEN
         Result := True;
      ELSE
         Result := False;
      END IF;
      RETURN Result;
   END Ask_Again;

   PROCEDURE Get_Number (
         Line   : IN     Y_Pos;
         Length : IN     Positive;
         Number :    OUT String) IS
      Index     : Positive  := 1;
      Key       : Character := 'X';
      X_Start   : X_Pos     := (X_Pos'Last - Length + 1) / 2;
      X_Current : X_Pos     := X_Start;
   BEGIN
      Number := (OTHERS => '.');
      Set_Foreground (Light_Cyan);
      Goto_XY (X_Start, Line);
      FOR I IN 1 .. Length LOOP
         Put ('.');
      END LOOP;
      Goto_XY (X_Current, Line);
      Set_Cursor (True);
      WHILE Key /= ASCII.CR LOOP
         Key := Get_Key;
         IF Key = ASCII.NUL THEN
            Key := Get_Key;
            CASE Key IS
               WHEN Key_Delete =>
                  Number (Index) := '.';
                  Put ('.');
                  Goto_XY (X_Current, Line);
               WHEN Key_Right  =>
                  IF Index < Length THEN
                     Index := Index + 1;
                     X_Current := X_Current + 1;
                     Goto_XY (X_Current, Line);
                  END IF;
               WHEN Key_Left  =>
                  IF Index > 1 THEN
                     Index := Index - 1;
                     X_Current := X_Current - 1;
                     Goto_XY (X_Current, Line);
                  END IF;
               WHEN OTHERS     =>
                  Bleep;
            END CASE;
         ELSE
            CASE Key IS
               WHEN ASCII.CR   =>
                  NULL;
               WHEN ASCII.BS   =>
                  IF Index > 1 THEN
                     Index := Index - 1;
                     X_Current := X_Current - 1;
                     Number (Index) := '.';
                     Goto_XY (X_Current, Line);
                     Put ('.');
                     Goto_XY (X_Current, Line);
                  END IF;
               WHEN '0' .. '9' =>
                  Number (Index) := Key;
                  Put (Key);
                  Index := Index + 1;
                  X_Current := X_Current + 1;
                  IF Index > Length THEN
                     Index := Index - 1;
                     X_Current := X_Current - 1;
                     Goto_XY (X_Current, Line);
                     Index := Length;
                  END IF;
               WHEN OTHERS     =>
                  Bleep;
            END CASE;
         END IF;
      END LOOP;
      Set_Cursor (False);
   END Get_Number;

   PROCEDURE Remove_Spaces (
         Number : IN OUT String) IS
      I : Natural := Index (Number, ".");
   BEGIN
      WHILE I /= 0 LOOP
         Delete (Number, I, I);
         I := Index (Number, ".");
      END LOOP;
   END Remove_Spaces;

   FUNCTION Number_Length (
         Number : String)
     RETURN Natural IS
      Result : Natural := Index (Number, " ");
   BEGIN
      IF Result /= 0 THEN
         Result := Result - 1;
      END IF;
      RETURN Result;
   END Number_Length;

   PROCEDURE Display_Centered (
         Line : IN     Y_Pos;
         Text : IN     String) IS
      X : X_Pos := (X_Pos'Last - Text'Length) / 2;
   BEGIN
      Goto_XY (X, Line);
      Put (Text);
   END Display_Centered;

   FUNCTION Find_Company (
         Number : String)
     RETURN Card_Company IS
      Company : Card_Company := Unknown;
   BEGIN
      Find_Card:
         FOR I IN Card_Table'RANGE LOOP
         IF Number (1 .. Card_Table(I).Prefix_Length) =
               Card_Table(I).Prefix_String (1 .. Card_Table(I).
               Prefix_Length)
               THEN
            Company := Card_Table(I).Card;
            EXIT Find_Card;
         END IF;
      END LOOP Find_Card;
      RETURN Company;
   END Find_Company;

   FUNCTION Validate (
         S : IN     String;
         N : IN     Natural)
     RETURN Boolean IS
      Temp     : Natural;
      Checksum : Natural := 0;
      Double   : Boolean := False;
   BEGIN
      FOR I IN REVERSE 1 .. N LOOP
         Temp := Integer'Value (S(I .. I));
         IF Double THEN
            Temp := 2 * Temp;
            IF Temp > 9 THEN
               Temp := Temp - 9;
            END IF;
         END IF;
         Checksum := Checksum + Temp;
         Double := NOT Double;
      END LOOP;
      RETURN (Checksum mod 10) = 0;
   END Validate;

BEGIN
   WHILE Again = True LOOP
      Set_Cursor (False);
      Display_Banner;
      Get_Number (10, Max_Length, Number);
      Remove_Spaces (Number);
      Length := Number_Length (Number);
      Company := Find_Company (Number);
      CASE Company IS
         WHEN Unknown    =>
            Set_Foreground (White);
            Set_Background (Red);
            Display_Centered (12, "UNRECOGNIZED CARD NUMBER");
         WHEN Mastercard =>
            IF Length /= 16 THEN
               Set_Foreground (White);
               Set_Background (Red);
               Display_Centered (12, "UNRECOGNIZED MASTERCARD NUMBER");
            ELSIF Validate (Number, Length) = False THEN
               Set_Foreground (White);
               Set_Background (Red);
               Display_Centered (12, "INVALID MASTERCARD");
            ELSE
               Set_Foreground (White);
               Set_Foreground (Green);
               Display_Centered (12, "VALID MASTERCARD");
            END IF;
         WHEN Visa       =>
            IF Length /= 13 AND Length /= 16 THEN
               Set_Foreground (White);
               Set_Background (Red);
               Goto_XY (28, 12);
               Put ("UNRECOGNIZED VISA NUMBER");
            ELSIF Validate (Number, Length) = False THEN
               Set_Foreground (White);
               Set_Background (Red);
               Display_Centered (12, "INVALID VISA CARD");
            ELSE
               Set_Foreground (White);
               Set_Background (Green);
               Display_Centered (12, "VALID VISA CARD");
            END IF;
         WHEN Amex       =>
            IF Length /= 15 THEN
               Set_Foreground (White);
               Set_Background (Red);
               Goto_XY (22, 12);
               Put ("UNRECOGNIZED AMERICAN EXPRESS NUMBER");
            ELSIF Validate (Number, Length) = False THEN
               Set_Foreground (White);
               Set_Background (Red);
               Display_Centered (12, "INVALID AMERICAN EXPRESS CARD");
            ELSE
               Set_Foreground (White);
               Set_Background (Green);
               Display_Centered (12, "VALID AMERICAN EXPRESS CARD");
            END IF;
         WHEN Diners     =>
            IF Length /= 14 THEN
               Set_Foreground (White);
               Set_Background (Red);
               Goto_XY (26, 12);
               Put ("UNRECOGNIZED DINERS NUMBER");
            ELSIF Validate (Number, Length) = False THEN
               Set_Foreground (White);
               Set_Background (Red);
               Display_Centered (12, "INVALID DINERS CARD");
            ELSE
               Set_Foreground (White);
               Set_Background (Green);
               Display_Centered (12, "VALID DINERS CARD");
            END IF;
      END CASE;
      Again := Ask_Again;
   END LOOP;
   Set_Foreground (Gray);
   Clear_Screen;
   Goto_XY;
   Set_Cursor (True);
END Demo;
