/*!
 * @file HSyntaxHighlighter.h
 * File contains declaration of HSyntaxHighlighter class
 * @author Krzysztof Wesołowski
 */
#ifndef HSYNTAXHIGHLIGHTER_H_
#define HSYNTAXHIGHLIGHTER_H_

#include "QtGui/QSyntaxHighlighter"
#include "QtGui/QMessageBox"

#include "HSyntaxHighlighterSettings.h"
/*!
 * @class QSyntaxHighlighter
 * @brief Generic class for all Qt's syntax highlighters
 */

/*!
 * @brief Syntax highlighter
 * This class implements simple Syntax Highlighter,
 * based on rules using QRegExp form matching parts of code to highlight,
 * and QTextCharFormat for code formating. It developed on basis of Qt's QSyntaxHighlighter,
 * which provide foundations for all Syntax Highlighting classes
 * It store all of it settings in settings
 */
class HSyntaxHighlighter: public QSyntaxHighlighter
{
    Q_OBJECT

private:
    QVector<slFormatingRule> sRules; ///<Single line rules used to highlight (cached from settings)
    QVector<mlFormatingRule> mRules; ///<Multiple line rules used to highlight (cached from settings)

    /*!
     * Variable storing all HSyntaxHighlighter settings,
     * used to generate sRules and mRules,
     * easily accessible via config() method.
     */
    HSyntaxHighlighterSettings settings;
    /*!
     * Method for (re)caching rules, must me called after settings are modified.
     */
    void updateRules();

public:
    /*!
     * @brief Constructor
     * Loads default configuration into settings.
     * @param parent Pointers to owners object
     */
    HSyntaxHighlighter(QObject * parent);

    /*!
     * @brief Constructor
     * Loads default configuration into settings.
     * @param parent Pointers to owners object and initial QTextDocument on which highlighting is applied
     */
    HSyntaxHighlighter(QTextDocument * parent);

    /*!
     * @brief Constructor
     * Loads default configuration into settings.
     * @param parent Pointers to owners object and QTextEdit which holds highlighted QTextDocument
     */
    HSyntaxHighlighter(QTextEdit * parent);

    /*!
     * @brief Highlighting function
     * After document content is changed this function is call for all lines,
     * from the current changed line to the end of document. Any line is separate block,
     * so storing current state is needed for multiple line code blocks for example longer comments.
     * Implementing this virtual QSyntaxHighlighter Method is needed to make it works
     * @param text Text of a single line, that is send to this Highlighter.
     */
    void highlightBlock(const QString &text);
    /*!
     * @brief Destructor
     * Since class store all data on stack, destructor only saves default AKA last used settings to file.
     */
    ~HSyntaxHighlighter();

    /*!
     * @brief Settings pointer.
     * @return Pointer to be used for calling method to modify settings.
     */
    HSyntaxHighlighterSettings * config()
    {
        return &settings;
    }

public slots:

    /*!
     * Slot used to launch settings dialog for stored configuration
     */
    void editConfig();

    /*!
     * Slot used to launch Save Settings dialog for stored configuration
     */
    void saveConfig() const;

    /*!
     * Slot used to launch Load Settings dialog for stored configuration
     */
    void loadConfig();

};

#endif /* HSYNTAXHIGHLIGHTER_H_ */
