/*!
* @file HSyntaxHighlighterDialog.h
* File contains declaration of HSyntaxHighlighterDialog class
* @author Krzysztof Wesołowski
*/
#ifndef HSYNTAXHIGHLIGHTERSETTINGSDIALOG_H
#define HSYNTAXHIGHLIGHTERSETTINGSDIALOG_H

#include <QtGui/QDialog>
#include <QtGui/QLineEdit>
#include <QtGui/QRadioButton>
#include <QtGui/QInputDialog>
#include <QtGui/QMessageBox>

#include "ui_HSyntaxHighlighterDialog.h"
#include "HSyntaxHighlighterSettings.h"
#include "HSyntaxHighlighterNewBlock.h"
#include "HighlightBlock.h"
/*!
* @class QDialog
* @brief Base class for all dialog windows/widgets, providing exec(), accepted() and rejected()
*/

/*!
* @brief Class implementing Dialog,
* which is used to edit HSyntaxHighlighterSettings,
* it contains list of all HighlightBlock s and QGroupBox with editing panel.
*/
class HSyntaxHighlighterDialog : public QDialog
{
    Q_OBJECT
private:
    /*!
    * Copying constructor.
    * Provided to ensure object ISN'T copied.
    */
    HSyntaxHighlighterDialog(const HSyntaxHighlighterDialog&);

    Ui::HSyntaxHighlighterDialogClass userInterface; ///< Qt Designer Generated basic UI for this dialog
    HSyntaxHighlighterSettings settings; ///< Local copy of settings, user edit it, and any time it can be read via getSettings()
    /*!
    * Function filling userInterface.HighlightBlocks QListWidget with all HighlightBlocks pointed by  allBlocks
    */
    void constructBlockList();
private slots:
/*!
* @brief Slot called to change edited item, after calling it destroys current userInterface.groupBox
* and then replaces it with new appropriate  HBSettingPanel
* @param _name of new HighlightBlock to edit now.
*/
void editedItemChanged(const QString& _name);
/*!
* @brief adds new block
*/
void addNewItem()
{
    HSyntaxHighlighterNewBlock dialog(&settings,this);
    if (dialog.exec()==QDialog::Accepted)
        {
            constructBlockList();
            QListWidgetItem * tmp= userInterface.HighlightBlocks->findItems(dialog.getName(),Qt::MatchFixedString).at(0);
            userInterface.HighlightBlocks->setCurrentItem(tmp);
        }
}

/*!
* @brief removes selected block
*/
void removeCurrentItem()
{
    if(userInterface.HighlightBlocks->count() >0)
        {
            QString name=userInterface.HighlightBlocks->currentItem()->text();
            if (QMessageBox::Ok==QMessageBox::question(this,"Deleting block","Are you sure to delete block: "+name+"?",QMessageBox::Ok | QMessageBox::Cancel))
                {
                    settings.removeBlock(name);
                    constructBlockList();
                }
        }
}

/*!
* @brief renames current block
*/
void renameCurrentItem()
{
    if(userInterface.HighlightBlocks->count() >0)
        {
            QString curr=userInterface.HighlightBlocks->currentItem()->text();
            bool ok;
            QString text = QInputDialog::getText(this,"Renaming highlight block","Enter a new name for an "+curr+" highlight block"
                    , QLineEdit::Normal,
                    curr, &ok);
            if (ok && !text.isEmpty())
                if(curr!=text)
                    if(settings.blockExists(text))
                        {
                            QMessageBox::warning(this,"Error while renaming","Renaming "+curr+" to "+" "+text+" failed, because name must be unique");
                        }
                    else
                        {
                            settings.renameBlock(curr,text);
                            constructBlockList();
                        }
        }
}
public:
    /*!
    * @brief Constructor
    * creates UI and initialize  slBlocks and  mlBlocks with NULL
    * @param parent Pointer to parent QWidget, used by QDialog constructor
    */
    HSyntaxHighlighterDialog(QWidget *parent = 0);
    /*!
    * @brief Destructor
    * If settings were set, it deletes HighlightBlocks pointed by  slBlocks and  mlBlocks
    */
    ~HSyntaxHighlighterDialog();

    /*!
    * @brief settings getter
    * This function modify current settings, by copying HighlightBlocks stored,
    * and possibly modified in  slBlocks and  mlBlocks into settings
    * @return Possibly modified  HSyntaxHighlighter settings.
    */
    HSyntaxHighlighterSettings getSettings();

    /*!
    * @brief Set up initial settings
    * This function is used to setup initial settings of  HSyntaxHighlighterDialog,
    * it should be used before executing dialog, if you forget about it behavior may be unexpected
    * @param _settings Initial settings to store in  settings
    */
    void setSettings(HSyntaxHighlighterSettings _settings);

};

#endif // HSYNTAXHIGHLIGHTERSETTINGSDIALOG_H
