/*!
* @file HSyntaxHighlighterSettings.h
* File contains declaration of HSyntaxHighlighterSettings class
* @author Krzysztof Wesołowski
*/

#ifndef HSYNTAXHIGHLIGHTERSETTINGS_H_
#define HSYNTAXHIGHLIGHTERSETTINGS_H_
#include <QtCore/QString>
#include <QtCore/QVector>
#include <QtGui/QTextCharFormat>
#include <QtCore/QCoreApplication>

#include "HighlightBlock.h"
/*!
* @brief Enum containing used Single Lines highlights
* It's intended to make any tables/QVector containing
* highlight rules to be more intuitive, with indexes form this enum
*/

struct slFormatingRule
{
    QRegExp exp; ///< Matching Regular Expression
    QTextCharFormat format; ///<Format applied to matched text
};
/*!
* @brief Multiple lines formating rule
* Simple struct designed to hold matching starting and ending regular expression
* and format of matched text together. Text matched by this rule can span over multiple lines.
*/
struct mlFormatingRule
{
    QRegExp beginingExp;   ///<Expression matching beginning of highlighted block
    QRegExp endingExp;     ///<Expression matching ending of highlighted block
    QTextCharFormat format;///<Format applied to matched text
};

/*!
* @brief Highlighting Settings
* This class holds all Highlighting settings, used by HSyntaxHighlighter
* It is used to generate formating rules.
*  slFormatingRule
*  mlFormatingRule
*  HSyntaxHighlighterDialog
*/
class HSyntaxHighlighterSettings
{
    /*!
    * This class is friend, because HSyntaxHighlighterDialog is a visual interface,
    * and avoiding this friend will cause breaking Settings encapsulation,
    * with getters and setters given to everyone instead of this one class.
    */
    friend class HSyntaxHighlighterDialog;
private:
    QMap<QString,SingleLineHighlightBlock> singleLine; ///<Map of single lines block indexed by names
    QMap<QString,MultiLinesHighlightBlock> multiLines; ///<Map of multiple lines block indexed by names

    /*!
    * @brief variable indicating if settings were modified since last caching,
    * indicated by call to cached()
    */
    bool modified;

public:
    /*!
    * @brief Function indicating if settings were changed, so user should opdate any cached info
    * @return True if settings were modified.
    */
    bool changed() const
    {
        return modified;
    }

    /*!
    * @brief Function that indicates blocks were cached cached
    * after is called changed() will return false until next modification.
    */
    void cached()
    {
        modified=false;
    }
    /*!
    * @brief Functions saves configuration to file
    * This function is used to store all highlighter settings,
    * such as matching patterns and formatting info selected ini file
    * @param _fileName complete path to ini file, where settings should be stored. If no specified, Applications directory and name default.ini are assumed.
    */
    void saveConfig(QString _fileName=QCoreApplication::applicationDirPath()+"/default.ini") const;

    /*!
    * @brief Functions loads configuration from file
    * This function is used to load all highlighter settings,
    * such as matching patterns and formatting form selected ini file
    * @param _fileName complete path to ini file, where settings should be stored. If no specified, Applications directory and name default.ini are assumed.
    */
    void loadConfig(QString _fileName=QCoreApplication::applicationDirPath()+"/default.ini");

    /*!
    * @brief Basic constructor of settings class
    * It only calls setupHighlightBlocks()
    */
    HSyntaxHighlighterSettings();
    /*!
    * @brief Destructor
    * We don't use any raw pointers here, so he has no job.
    */
    ~HSyntaxHighlighterSettings(){};
    /*!
    * @brief Generates single line rules
    * This function generates single line rules for  HSyntaxHighlighter
    * @return QVector of  slFormatingRule class rules.
    */
    QVector<slFormatingRule> getSingleLineRules() const;

    /*!
    * @brief Generates multiple line rules
    * This function generates multiple line rules for  HSyntaxHighlighter
    * @return QVector of  mlFormatingRule class rules.
    */
    QVector<mlFormatingRule> getMultiLinesRules() const;

    /*!
    * @brief adds new empty highlighting block to be filled by user.
    * @param _name name of the new block
    * @param _type
    */
    void addBlock(const QString& _name,HighlightBlock::HighlighType _type=HighlightBlock::sl)
    {
        if (!blockExists(_name))
            {
                if (_type==HighlightBlock::sl)
                    {
                        SingleLineHighlightBlock block;
                        block.setName(_name);
                        singleLine.insert(_name,block);
                    }
                else if (_type==HighlightBlock::ml)
                    {
                        MultiLinesHighlightBlock block;
                        block.setName(_name);
                        multiLines.insert(_name,block);
                    }
            }
        modified=true;
    }

    /*!
    * @brief checks if block exists
    * @param _name name of checked block
    * @return true if exists, false otherwise
    */
    bool blockExists(const QString& _name) const
    {
        return singleLine.contains(_name)||multiLines.contains(_name);
    }
    /*!
    * @brief Removes block if it exists
    * @param _name Name of block to be removed
    */
    void removeBlock(const QString& _name)
    {
        if (singleLine.contains(_name))
            singleLine.remove(_name);
        else if (multiLines.contains(_name))
            multiLines.remove(_name);

        modified=true;
    }
    void renameBlock(const QString& _oldName,const QString& _newName)
    {
        if (singleLine.contains(_oldName))
            {
                SingleLineHighlightBlock temp=singleLine.take(_oldName);
                temp.setName(_newName);
                singleLine.insert(_newName,temp);
            }
        else if (multiLines.contains(_oldName))
            {
                MultiLinesHighlightBlock temp=multiLines.take(_oldName);
                temp.setName(_newName);
                multiLines.insert(_newName,temp);
            }
        modified=true;
    }
};

#endif /* HSYNTAXHIGHLIGHTERSETTINGS_H_ */
