/*!
 * @file HSyntaxHighlighter.cpp
 * File contains implementation of HSyntaxHighlighter methods
 * @author Krzysztof Wesołowski
 */

#include <QtGui/QSyntaxHighlighter>
#include <QtGui/QFileDialog>

#include "HSyntaxHighlighter.h"
#include "HSyntaxHighlighterDialog.h"

HSyntaxHighlighter::HSyntaxHighlighter(QObject * parent) : QSyntaxHighlighter(parent)
{
    settings.loadConfig();
}
HSyntaxHighlighter::HSyntaxHighlighter(QTextDocument * parent) : QSyntaxHighlighter(parent)
{
    settings.loadConfig();
}
HSyntaxHighlighter::HSyntaxHighlighter(QTextEdit * parent) : QSyntaxHighlighter(parent)
{
    settings.loadConfig();
}

void HSyntaxHighlighter::highlightBlock(const QString &text)
{
    if(settings.changed())
        updateRules();
    foreach (slFormatingRule rule, sRules) {
        int index = text.indexOf(rule.exp);
        while (index >= 0) {
            int length = rule.exp.matchedLength();
            if(length==0)
                break;
            setFormat(index, length, rule.format);
            index = text.indexOf(rule.exp, index + length);
        }
    }

    for (int i = 0; i < mRules.size(); ++i) {
        setCurrentBlockState(0);
        int startIndex = 0;
        if (previousBlockState() != (i+1)) //we wasn't in this kind of Block
            startIndex = text.indexOf(mRules[i].beginingExp); //find if we are in it now

        while (startIndex >= 0)
            {
                int endIndex = text.indexOf(mRules[i].endingExp, startIndex); //find end of this HighlightBlock
                int length;
                if (endIndex == -1) { //if we just started, or continue block, apply formatting
                    setCurrentBlockState(i+1);
                    length = text.length() - startIndex;
                } else {
                    length = endIndex - startIndex
                    + mRules[i].endingExp.matchedLength();
                }
                setFormat(startIndex, length, mRules[i].format);
                startIndex = text.indexOf(mRules[i].beginingExp,
                        startIndex + length);
                if(length==0)
                    break;
            }
    }
}


HSyntaxHighlighter::~HSyntaxHighlighter()
{
    settings.saveConfig();
}

void HSyntaxHighlighter::updateRules()
{
    sRules=settings.getSingleLineRules();
    mRules=settings.getMultiLinesRules();
    settings.cached();
}


void HSyntaxHighlighter::editConfig()
{

    HSyntaxHighlighterDialog configDialog;
    configDialog.setSettings(settings);
    configDialog.exec();
    if(configDialog.result()==QDialog::Accepted)
        settings=configDialog.getSettings();
    this->updateRules();
}
void HSyntaxHighlighter::saveConfig() const
{
    QString fileName=
        QFileDialog::getSaveFileName(0,"Save highlighting settings",QCoreApplication::applicationDirPath(),"Config Files (*.ini)");
    if(!fileName.isEmpty())
        settings.saveConfig(fileName);
}
void HSyntaxHighlighter::loadConfig()
{
    QString fileName=
        QFileDialog::getOpenFileName(0,"Load highlighting settings",QCoreApplication::applicationDirPath(),"Config Files (*.ini)");
    if(!fileName.isEmpty())
        settings.loadConfig(fileName);
    this->updateRules();
}
