/*!
 * @file HSyntaxHighlighterSettings.cpp
 * @brief Syntax Highlighter Settings implementation files.
 * File contains implementation of methods form class  HSyntaxHighlighterSettings
 * @author Krzysztof Wesołowski
 */

#include <QtCore/QSettings>

#include "HSyntaxHighlighterSettings.h"

HSyntaxHighlighterSettings::HSyntaxHighlighterSettings()
{
    setupHighlightBlocks();
    modified=true;
}

void HSyntaxHighlighterSettings::setupHighlightBlocks()
{
    singleLine.resize(5);
    singleLine[SingleLineCommment].setDescription("Simple one line comment, filling only one line");
    singleLine[SingleLineCommment].setName("Single-line comment");

    singleLine[Keyword].setDescription("Keyword, a word reserved for special purposes");
    singleLine[Keyword].setName("Keyword");

    singleLine[TextConstant].setDescription("Character constant, containing a single character or whole string ");
    singleLine[TextConstant].setName("Character constant");

    singleLine[NumberConstant].setDescription("Number constant, with possible formating/extra info characters");
    singleLine[NumberConstant].setName("Number constant");

    singleLine[PreprocessorDirective].setDescription("Preprocessor directive, kind of special code interpreted and used at compilation time");
    singleLine[PreprocessorDirective].setName("Preprocessor directive");

    multiLines.resize(1);
    multiLines[MultiLinesComment].setDescription("Longer comment, possibly spanning multiple lines, with specified beginning and ending");
    multiLines[MultiLinesComment].setName("Multiple-Line comment");
}




void HSyntaxHighlighterSettings::saveConfig(QString fileName) const
{
    QSettings settings(fileName,QSettings::IniFormat);
    foreach(SingleLineHighlightBlock block,singleLine)
    {
        settings.setValue(block.getName()+"/pattern",block.getPattern());

        QFont font=block.getFont();
        settings.setValue(block.getName()+"/Font Family",font.family());
        settings.setValue(block.getName()+"/Font Size",font.pointSize());
        settings.setValue(block.getName()+"/Font Italic",font.italic());
        settings.setValue(block.getName()+"/Font Bold",font.bold());
        settings.setValue(block.getName()+"/Font Underline",font.underline());
        settings.setValue(block.getName()+"/Font StrikeOut",font.strikeOut());

        QString hexColor=QString::number(block.getForegroundColor().rgb(),16);
        settings.setValue(block.getName()+"/Foreground Color",hexColor);
        hexColor=QString::number(block.getBackgroundColor().rgb(),16);
        settings.setValue(block.getName()+"/Background Color",hexColor);
    }

    foreach(MultiLinesHighlightBlock block,multiLines)
    {
        settings.setValue(block.getName()+"/starting pattern",block.getStartingPattern());
        settings.setValue(block.getName()+"/ending pattern",block.getEndingPattern());

        QFont font=block.getFont();
        settings.setValue(block.getName()+"/Font Family",font.family());
        settings.setValue(block.getName()+"/Font Size",font.pointSize());
        settings.setValue(block.getName()+"/Font Italic",font.italic());
        settings.setValue(block.getName()+"/Font Bold",font.bold());
        settings.setValue(block.getName()+"/Font Underline",font.underline());
        settings.setValue(block.getName()+"/Font StrikeOut",font.strikeOut());

        QString hexColor=QString::number(block.getForegroundColor().rgb(),16);
        settings.setValue(block.getName()+"/Foreground Color",hexColor);
        hexColor=QString::number(block.getBackgroundColor().rgb(),16);
        settings.setValue(block.getName()+"/Background Color",hexColor);
    }
}

void HSyntaxHighlighterSettings::loadConfig(QString  fileName)
{

    QSettings settings(fileName,QSettings::IniFormat);
    for (int i = 0; i < singleLine.size(); ++i)
        {
            SingleLineHighlightBlock& block=singleLine[i];
            QString w=settings.value(block.getName()+"/pattern").toString();

            block.setPattern(w);

            QFont font;
            font.setFamily(settings.value(block.getName()+"/Font Family").toString());
            int pointSize=settings.value(block.getName()+"/Font Size").toInt();
            font.setPointSize(pointSize> 0 ? pointSize : 8);
            font.setItalic(settings.value(block.getName()+"/Font Italic").toBool());
            font.setBold(settings.value(block.getName()+"/Font Bold").toBool());
            font.setUnderline(settings.value(block.getName()+"/Font Underline").toBool());
            font.setStrikeOut(settings.value(block.getName()+"/Font StrikeOut").toBool());
            block.setFont(font);

            QColor color=QColor::fromRgb(settings.value(block.getName()+"/Foreground Color").toString().toUInt(0,16));
            block.setForegroundColor(color);
            color=QColor::fromRgb(settings.value(block.getName()+"/Background Color").toString().toUInt(0,16));
            block.setBackgroundColor(color);
        }

    for (int i = 0; i < multiLines.size(); ++i) {

        MultiLinesHighlightBlock& block=multiLines[i];
        block.setStartingPattern(settings.value(block.getName()+"/starting pattern").toString());
        block.setEndingPattern(settings.value(block.getName()+"/ending pattern").toString());

        QFont font;
        font.setFamily(settings.value(block.getName()+"/Font Family").toString());
        int pointSize=settings.value(block.getName()+"/Font Size").toInt();
        font.setPointSize(pointSize>0 ? pointSize : 8);
        font.setItalic(settings.value(block.getName()+"/Font Italic").toBool());
        font.setBold(settings.value(block.getName()+"/Font Bold").toBool());
        font.setUnderline(settings.value(block.getName()+"/Font Underline").toBool());
        font.setStrikeOut(settings.value(block.getName()+"/Font StrikeOut").toBool());
        multiLines[i].setFont(font);

        QColor color=QColor::fromRgb(settings.value(block.getName()+"/Foreground Color").toString().toUInt(0,16));
        block.setForegroundColor(color);
        color=QColor::fromRgb(settings.value(block.getName()+"/Background Color").toString().toUInt(0,16));
        block.setBackgroundColor(color);
    }
    modified=true;
}
QVector<mlFormatingRule> HSyntaxHighlighterSettings::getMultiLinesRules() const
{
    mlFormatingRule rule;
    QVector<mlFormatingRule> result;

    foreach(MultiLinesHighlightBlock block,multiLines)
    {
        rule.beginingExp=block.getStartingExp();
        rule.endingExp=block.getEndingExp();
        rule.format=block.getFormat();
        result.append(rule);
    }
    return result;
}
QVector<slFormatingRule> HSyntaxHighlighterSettings::getSingleLineRules() const
{
    slFormatingRule rule;
    QVector<slFormatingRule> result;

    foreach(SingleLineHighlightBlock block,singleLine)
    {
        rule.exp=block.getExp();
        rule.format=block.getFormat();
        result.append(rule);
    }
    return result;
}



