/*!
 * @file HSyntaxHighlighterSettings.h
 * File contains declaration of HSyntaxHighlighterSettings class
 * @author Krzysztof Wesołowski
 */

#ifndef HSYNTAXHIGHLIGHTERSETTINGS_H_
#define HSYNTAXHIGHLIGHTERSETTINGS_H_
#include <QtCore/QString>
#include <QtCore/QVector>
#include <QtGui/QTextCharFormat>
#include <QtCore/QCoreApplication>

#include "HighlightBlock.h"
/*!
 * @brief Enum containing used Single Lines highlights
 * It's intended to make any tables/QVector containing
 * highlight rules to be more intuitive, with indexes form this enum
 */
enum slHighlightType {
	Keyword,              //!< Keyword
	TextConstant,         //!< TextConstant
	NumberConstant,       //!< NumberConstant
	PreprocessorDirective,//!< PreprocessorDirective
	SingleLineCommment    //!< SingleLineCommment
};
/*!
 * @brief Enum containing used Multiple Lines highlights
 * It's intended to make any tables/QVector containing
 * highlight rules to be more intuitive, with indexes form this enum
 */
enum mlHighlightType {
	MultiLinesComment      //!< MultiLinesComment
};
/*!
 * @brief Single line formating rule
 * Simple struct designed to hold Matching QRegExp and format of matched text together
 */
struct slFormatingRule
{
	QRegExp exp; ///< Matching Regular Expression
	QTextCharFormat format; ///<Format applied to matched text
};
/*!
 * @brief Multiple lines formating rule
 * Simple struct designed to hold matching starting and ending regular expression
 * and format of matched text together. Text matched by this rule can span over multiple lines.
 */
struct mlFormatingRule
{
	QRegExp beginingExp;   ///<Expression matching beginning of highlighted block
	QRegExp endingExp;     ///<Expression matching ending of highlighted block
	QTextCharFormat format;///<Format applied to matched text
};

/*!
 * @brief Highlighting Settings
 * This class holds all Highlighting settings, used by HSyntaxHighlighter
 * It is used to generate formating rules.
 *  slFormatingRule
 *  mlFormatingRule
 *  HSyntaxHighlighterDialog
 */
class HSyntaxHighlighterSettings
{
    /*!
     * This class is friend, because HSyntaxHighlighterDialog is a visual interface,
     * and avoiding this friend will cause breaking Settings encapsulation,
     * with getters and setters given to everyone instead of this one class.
     */
    friend class HSyntaxHighlighterDialog;
private:
    QVector<SingleLineHighlightBlock> singleLine; ///<Vector of single lines block, contains SingleLineHighlightBlock
    QVector<MultiLinesHighlightBlock> multiLines; ///<Vector of multiple lines block, contains MultiLinesHighlightBlock
    /*!
     * @brief Basic block setup
     * Function change size of Vectors storing Highlighting blocks, and then fills some basic info,
     * for each item: name and description.
     *  singleLine
     *  multiLines
     *  HighlightBlock::name
     *  HighlightBlock::description
     */
    void setupHighlightBlocks();
    /*!
     * @brief variable indicating if settings were modified since last call to
     * getSingleLineRules or  getMultiLinesRules
     */
    bool modified;

public:
    /*!
     * @brief Function indicating if settings were changed, so user should opdate any cached info
     * @return True if settings were modified.
     */
    bool changed() const
    {
        return modified;
    }

    /*!
     * @brief Function that indicates blocks were cached cached
     * after is called changed() will return false until next modification.
     */
    void cached()
    {
        modified=false;
    }
    /*!
     * @brief Functions saves configuration to file
     * This function is used to store all highlighter settings,
     * such as matching patterns and formatting info selected ini file
     * @param fileName complete path to ini file, where settings should be stored. If no specified, Applications directory and name default.ini are assumed.
     */
    void saveConfig(QString fileName=QCoreApplication::applicationDirPath()+"/default.ini") const;

    /*!
     * @brief Functions loads configuration from file
     * This function is used to load all highlighter settings,
     * such as matching patterns and formatting form selected ini file
     * @param fileName complete path to ini file, where settings should be stored. If no specified, Applications directory and name default.ini are assumed.
     */
    void loadConfig(QString fileName=QCoreApplication::applicationDirPath()+"/default.ini");

    /*!
     * @brief Basic constructor of settings class
     * It only calls setupHighlightBlocks()
     */
    HSyntaxHighlighterSettings();
    /*!
     * @brief Destructor
     * We don't use any raw pointers here, so he has no job.
     */
    ~HSyntaxHighlighterSettings(){};
    /*!
     * @brief Generates single line rules
     * This function generates single line rules for  HSyntaxHighlighter
     * @return QVector of  slFormatingRule class rules.
     */
    QVector<slFormatingRule> getSingleLineRules() const;

    /*!
     * @brief Generates multiple line rules
     * This function generates multiple line rules for  HSyntaxHighlighter
     * @return QVector of  mlFormatingRule class rules.
     */
    QVector<mlFormatingRule> getMultiLinesRules() const;

    /*!
     * @brief sets Font of specified Block
     * @param _type type of Highlighting block
     * @param _font font to use
     */
    void setFont(slHighlightType _type,const QFont& _font)
    {
        singleLine[_type].setFont(_font);
    }

    /*!
     * @brief sets Font of specified Block
     * @param _type type of Highlighting block
     * @param _font font to use
     */
    void setFont(mlHighlightType _type,const QFont& _font)
    {
        multiLines[_type].setFont(_font);
    }

    /*!
     * @brief sets Background Color of specified Block
     * @param _type type of Highlighting block
     * @param _color color to use
     */
    void setForegroundColor(slHighlightType _type,const QColor& _color)
    {
        singleLine[_type].setForegroundColor(_color);
    }

    /*!
     * @brief sets Background Color of specified Block
     * @param _type type of Highlighting block
     * @param _color color to use
     */
    void setForegroundColor(mlHighlightType _type,const QColor& _color)
    {
        multiLines[_type].setForegroundColor(_color);
    }

    /*!
     * @brief sets Background Color of specified Block
     * @param _type type of Highlighting block
     * @param _color color to use
     */
    void setBackgroundColor(slHighlightType _type,const QColor& _color)
    {
        singleLine[_type].setBackgroundColor(_color);
    }

    /*!
     * @brief sets Background Color of specified Block
     * @param _type type of Highlighting block
     * @param _color color to use
     */
    void setBackgroundColor(mlHighlightType _type,const QColor& _color)
    {
        multiLines[_type].setBackgroundColor(_color);
    }
};

#endif /* HSYNTAXHIGHLIGHTERSETTINGS_H_ */
