/*!
 * @file HighlightBlock.h
 * File contains declaration of HighlightBlock class
 * and its children MultiLinesHighlightBlock class, and SingleLineHighlightBlock
 * And since this classes are rather simple "containers",
 * all implementations of methods (no cpp file included)
 * @author Krzysztof Wesołowski
 */

#ifndef HIGHLIGHTBLOCK_H_
#define HIGHLIGHTBLOCK_H_
#include <QtCore/QString>
#include <QtGui/QTextCharFormat>

/*!
 * Enum holding possible typed of Highlight block, used by HighlightBlock::type() method
 */
enum HighlighType
{
    singleLine,//!< Indicates single line highlighting block
    multiLines //!< Indicates multiple lines highlighting block
};

/*!
 * @brief Class representing highlighting block with its name/description and formating
 * Class holds generic info for all available highlight block, its name and description,
 * used to store values in ini files and to build UI, and formatting aplied if block of code is detected.
 */
class HighlightBlock
{
private:
    QTextCharFormat format; ///< Formatting applied to matched block
    QString name; ///<User-friendly name of this block
    QString description; ///<Description of this block
public:

    /*!
     * Function for indicating a sub-type of HighlighBlock
     * @return Indicator from enum HighlighType
     */
    virtual HighlighType type()=0;
    /*!
     * Empty destructor because class doesn't allocate memory.
     */
    virtual ~HighlightBlock(){}

    /*!
     * @brief Formatting getter,
     * used for caching purposes
     * @return Formatting used by this HighlightBlock
     */
    QTextCharFormat getFormat() const
    {
        return format;
    }

    /*!
     * @brief Formatting font getter
     * provided for convenience, used for configuration purposes
     * @return currently used font
     */
    QFont getFont() const
    {
        return format.font();
    }

    /*!
     * @brief Formatting font setter
     * provided for convenience, used for configuration purposes
     * @param __font Font to be used in current format
     * @return currently used font
     */
    void setFont(const QFont& __font)
    {
        format.setFont(__font);
    }

    /*!
     * @brief Formatting foreground color getter
     * provided for convenience, used for configuration purposes
     * @return currently used foreground color
     */
    const QColor& getForegroundColor() const
    {
        return format.foreground().color();
    }

    /*!
     * @brief Formatting foreground color setter
     * provided for convenience, used for configuration purposes
     * @param color foreground color to be used
     */
    void setForegroundColor(const QColor& color)
    {
        format.setForeground(color);
    }

    /*!
     * @brief Formatting background color getter
     * provided for convenience, used for configuration purposes
     * @return currently used background color
     */
    const QColor& getBackgroundColor() const
    {
        return format.background().color();
    }

    /*!
     * @brief Formatting background color setter
     * provided for convenience, used for configuration purposes
     * @param color background color to be used
     */
    void setBackgroundColor(const QColor& color)
    {
        format.setBackground(color);
    }

    /*!
     * HighlightBlock name getter
     * @return user-friendly name of this block
     */
    QString getName() const
    {
        return name;
    }

    /*!
     * HighlightBlock name setter
     * @param _name new user-friendly name to be used
     */
    void setName(QString _name)
    {
        this->name = _name;
    }

    /*!
     * HighlightBlock description getter
     * @return user-friendly description of this block
     */
    QString getDescription() const
    {
        return description;
    }

    /*!
     * HighlightBlock description setter
     * @param _description new user-friendly description to be used
     */
    void setDescription(QString _description)
    {
        this->description = _description;
    }

};

/*!
 * @brief class providing Single Line Highlight block
 * (Highlighting block couldn't span over multiple lines)
 */
class SingleLineHighlightBlock : public HighlightBlock
{
private:
    QString pattern; ///< String pattern used for genereting QRegExp
public:
    /*!
     * @brief Creates Regular Expression from pattern
     * used for caching purposes
     * @return newly created RegularExpresion
     */
    QRegExp getExp() const
    {
        return QRegExp(pattern);
    }

    /*!
     * Text pattern setter
     * used for configuration purposes
     * @param _pattern new pattern to be used
     */
    void setPattern(QString _pattern) {pattern=_pattern;}

    HighlighType type()
    {
        return singleLine;
    }

    /*!
     * Text pattern getter
     * used for configuration purposes
     * @return current text pattern
     */
    QString getPattern() const
    {
        return pattern;
    }
};

/*!
 * @brief class providing Multiple Lines Highlight block
 * (Highlighting block which can span over multiple lines)
 */
class MultiLinesHighlightBlock : public HighlightBlock
{
private:
    QString startingPattern; ///< Pattern to match beginning of block
    QString endingPattern;   ///< Pattern to match ending of block
public:
    /*!
     * @brief Creates Regular Expression from startingPattern
     * used for caching purposes
     * @return newly created RegularExpresion
     */
    QRegExp getStartingExp() const
    {
        return QRegExp(startingPattern);
    }

    /*!
     * @brief Creates Regular Expression from endingPattern
     * used for caching purposes
     * @return newly created RegularExpresion
     */
    QRegExp getEndingExp() const
    {
        return QRegExp(endingPattern);
    }

    /*!
     * Text starting pattern setter
     * used for configuration purposes
     * @param _pattern new start pattern to be used
     */
    void setStartingPattern(QString _pattern)
    {
        startingPattern=_pattern;
    }

    /*!
     * Text ending pattern setter
     * used for configuration purposes
     * @param _pattern new end pattern to be used
     */
    void setEndingPattern(QString _pattern)
    {
        endingPattern=_pattern;
    }

    HighlighType type()
    {
        return multiLines;
    }

    /*!
     * Text starting pattern getter
     * used for configuration purposes
     * @return current start text pattern
     */
    QString getStartingPattern() const
    {
        return startingPattern;
    }

    /*!
     * Text ending pattern getter
     * used for configuration purposes
     * @return current end text pattern
     */
    QString getEndingPattern() const
    {
        return endingPattern;
    }
};
#endif /* HIGHLIGHTBLOCK_H_ */

