/*!
* \file ds1338.c
* \author Krzysztof Wesolowski
* \date September 09, 2009
*/

#include <util/twi.h>

#include "../twi_basics/twi_basics.h"
#include "ds1338.h"

/*!
 * Initiates uC to work with DS1338 twi interface.
 */
void ds1338_setup()
{
	twi_set_bitrate(TWI_BITRATE);
	twi_init();
}

/*!
 * Internal function, used to start and write specified pointer to DS1338
 * @param location Desired target memory location
 * @return DS1338 status code for use in other functions
 */
static ds1338_status ds1338_writePointer(uint8_t location)
{
	if(location<64)
	{
		switch (twi_start()) {
		case TW_START:
			//this is expected
			break;
		default:
			return BUS_ERROR;
		}

		switch (twi_send(DEVICE|WRITE)) {
		case TW_MT_SLA_ACK:
			//this is expected
			break;
		case TW_MT_SLA_NACK:
			return NO_DEVICE;
		default:
			return BUS_ERROR;
		}

		switch (twi_send(location)) {
		case TW_MT_DATA_ACK:
			//this is expected
			break;
		case TW_MT_DATA_NACK:
			return DEVICE_BUSY;
		default:
			return BUS_ERROR;
		}
		return NO_ERROR;
	}
	else
		return LOCATION_OUTOFBOUNDS;
}

/*!
 * Restarts TWI interface and send read adress
 * @return DS1338 status code for use in other functions
 */
static ds1338_status ds1338_restartToRead()
{
	switch (twi_start()) {
	case TW_REP_START:
		//this is expected
		break;
	default:
		return BUS_ERROR;
	}
	switch (twi_send(DEVICE|READ)) {
	case TW_MR_SLA_ACK:
		//this is expected
		break;
	case TW_MR_SLA_NACK:
		return NO_DEVICE;
	default:
		return BUS_ERROR;
	}
	return NO_ERROR;
}
/*!
 * Retrieves Time/Data data from RTC
 * @param data Pointer to target memory buffer
 * @return	DS1338 status code
 */
ds1338_status ds1338_readTimeDateControl(ds1338TDC * data)
{
	return ds1338_readBufferFromNVRAM(0x00,8,data->rawData);
}

/*!
 * Stores data in RTC - sets Time/Date
 * @param data Pointer to ds1338TDC structure
 * @return	DS1338 status code
 */
ds1338_status ds1338_writeTimeDateControl(ds1338TDC * data)
{
	return ds1338_writeBufferToNVRAM(0x00,8,data->rawData);
}

/*!
 * Writes specified data to RTC NVRAM
 * @param location 	Target starting location
 * @param lenght	Data lenght
 * @param data		Pointer to data
 * @return	DS1338 status code
 */
ds1338_status ds1338_writeBufferToNVRAM(uint8_t location,uint8_t lenght, uint8_t * data)
{
	ds1338_status status;
	if((status=ds1338_writePointer(location)) != NO_ERROR)
		return status;
	if(location+lenght >64)
		return RANGE_OUTOFBOUNDS;
	int i=0;
	do
	{
		if(twi_send(data[i++])!=TW_MT_DATA_ACK)
			return DEVICE_BUSY;
	}while(i<lenght);
	twi_stop();
	return NO_ERROR;
}

/*!
 * Read data from RTC NVRAM
 * @param location Starting location of readed data
 * @param lenght	Lenght of read data
 * @param data		Pointer where data we'll be stored
 * @return	DS1338 status code
 */
ds1338_status ds1338_readBufferFromNVRAM(uint8_t location,uint8_t lenght, uint8_t * data)
{
	ds1338_status status;
	if((status=ds1338_writePointer(location)) != NO_ERROR)
		return status;
	if(location+lenght >64)
		return RANGE_OUTOFBOUNDS;
	if((status=ds1338_restartToRead()) != NO_ERROR)
		return status;

	lenght--;
	int i=0;
	do
	{
		twi_receive(&data[i],lenght-i);
		i++;
	}while(i<=lenght);
	twi_stop();
	return NO_ERROR;
}

/*!
 * Writes signle byte to RTC NVRAM
 * @param location Target location
 * @param data	Data to be stored
 * @return DS1338 status code
 */
ds1338_status ds1338_writeByteToNVRAM(uint8_t location, uint8_t data)
{
	ds1338_status status;
	if((status=ds1338_writePointer(location)) != NO_ERROR)
		return status;

	if(twi_send(data)!=TW_MT_DATA_ACK)
		return DEVICE_BUSY;

	twi_stop();
	return NO_ERROR;

}
/*!
 * Read single byte from RTC NVRAM
 * @param location location of data in RTC
 * @param data Pointer where data we'll be stored
 * @return	DS1338 status code
 */
ds1338_status ds1338_readByteFromNVRAM(uint8_t location, uint8_t * data)
{
	ds1338_status status;
	if((status=ds1338_writePointer(location)) != NO_ERROR)
		return status;
	if((status=ds1338_restartToRead()) != NO_ERROR)
		return status;
	twi_receive(data,0);
	twi_stop();
	return NO_ERROR;
}
