/*!
 * \file twi_basics.c
 * \author Krzysztof Wesolowski
 * \date September 09, 2009
 */
#include "twi_basics.h"
#include <util/twi.h>

/*!
 * those bits should be unchanged while writing commands to TWCR, so we will simple write (TCWR & TW_CONF_MASK) | Command
 */
#define TW_CONF_MASK ((1<<TWWC) | (1<<TWEN) | (1<<TWIE) | (1<<1) )

#define TW_SEND_CMD(cmd) TWCR = (TWCR & TW_CONF_MASK ) | (cmd)			///<sends command leaving CONF bits unchanged

/*!
 * Waits until last TWI operation is completed
 */
void wait()
{
	while (!(TWCR & (1<<TWINT)));
}

/*!
 * Sends start condition
 * @return Status of TWI after operation extracted from TWSR
 */
twi_status twi_start()
{
	TW_SEND_CMD( (1<<TWINT) | (1<<TWSTA) );
	wait();
	return TW_STATUS;
}

/*!
 * Sends one byte over TWI interface
 * @param data Data to be send
 * @return Status of TWI after operation extracted from TWSR
 */
twi_status twi_send(uint8_t data)
{
	TWDR = data;
	TW_SEND_CMD( (1<<TWINT) );
	wait();
	return TW_STATUS;
}

/*!
 * Receive one byte over TWI interface
 * @param where Pointer to location where received data we'll be stored
 * @param ack	Set to nonzero to send ACK after receiving. Usually it continues transfer.
 * @return Status of TWI after operation extracted from TWSR
 */
twi_status twi_receive(uint8_t * where,uint8_t ack)
{
	if(ack)
		TW_SEND_CMD((1<<TWINT)|(1<<TWEA));
	else
		TW_SEND_CMD(1<<TWINT);
	wait();
	*where=TWDR;
	return TW_STATUS;
}

/*!
 * Sends STOP condition. Take a look at unusual busy-wait loop (Stop doesn't affect TWINT)
 */
void twi_stop()
{
	TW_SEND_CMD((1<<TWINT)|(1<<TWSTO));
	while ((TWCR & (1<<TWSTO)));
}
