/*
 * Graph.cpp
 *
 *  Created on: 2009-06-07
 *      Author: Wesoly
 */

#include "GraphVerticle.h"
#include "GraphEdge.h"
#include "Graph.h"
#include <cstdlib>

#include <QGraphicsSceneMouseEvent>
#include <QGraphicsItem>
#include <QDebug>
#include <QFile>
#include <QDataStream>



Graph::Graph()
{
    m_state=IdleEdit;
    m_line=NULL;
    setSceneRect(0,0,100,100);
}
/*!
* Returns edge between verticles (if not exist returns NULL)
* @param start Starting verticle
* @param end Ending verticle
* @return
*/
GraphEdge * Graph::edge(GraphVerticle * start,GraphVerticle * end)
{
    index tmp=index(start,end);
    if(m_edges.contains(tmp)) return m_edges[tmp];
    tmp=index(end,start);
    if(m_edges.contains(tmp)) return m_edges[tmp];
    return NULL;
}

void Graph::setNewEgdesWeightScheme(weightScheme scheme)
{
    m_weightScheme=scheme;
}

/*!
* Returns edge between verticles (if not exist returns NULL)
* @param start Starting verticle
* @param end Ending verticle
* @return
*/
const GraphEdge * Graph::edge(GraphVerticle * start,GraphVerticle * end) const
{
    index tmp=index(start,end);
    if(m_edges.contains(tmp)) return m_edges[tmp];
    tmp=index(end,start);
    if(m_edges.contains(tmp)) return m_edges[tmp];
    return NULL;
}
void Graph::remapAll()
{
    int i=0;
    foreach(GraphVerticle * vert,m_verticles)
    {
        vert->setSymbol('A'+i++);
    }
    foreach(GraphEdge * edge, m_edges)
    {
        edge->redraw();
    }
}

void Graph::changeState(CurrentState state)
{
    m_state=state;
    switch (state) {
        case IdleEdit:
            emit   stateChanged(trUtf8("Oczekiwanie - wybierz opcję..."));
            break;
        case AddingNewEdge:
            emit   stateChanged(trUtf8("Dodawanie krawędzi, przeciągnij aby dodać"));
            break;
        case AddingNewVerticle:
            emit   stateChanged(trUtf8("Dodawanie wierzchołków, kliknij LPM aby dodać"));
            break;
        case RemovingVerticle:
            emit   stateChanged(trUtf8("Usuwanie wierzchołków, kliknij LPM aby usunąć"));
            break;
        case RemovingEdge:
            emit   stateChanged(trUtf8("Usuwanie krawędzi, kliknij LPM aby usunąc"));
            break;
    }
}
void Graph::addVerticle()
{
    changeState(AddingNewVerticle);
    emit showExternInfo(trUtf8("Kliknij dwukrotnie aby dodać wierzchołek..."));
}

GraphVerticle * Graph::addVerticle(QPointF where)
{
    GraphVerticle * tmp = new GraphVerticle(this);
    tmp->setPos(where);
    m_verticles.append(tmp);
    addItem(tmp);
    remapAll();
    return tmp;
}

void Graph::removeVerticle()
{
    changeState(RemovingVerticle);
}

void Graph::removeVerticle(GraphVerticle * vert)
{
    QList<GraphEdge *> inc= incidental(vert);
    foreach(GraphEdge * edge, inc)
    {
        removeEdge(edge);
    }
    delete vert;
    m_verticles.removeOne(vert);
    remapAll();
}
void Graph::addEdge()
{
    changeState(AddingNewEdge);
}
/*!
* Checks if edge between two verticles exists
* @param start Starting verticle
* @param end Ending verticle
* @return True or false
*/
bool Graph::edgeExists(GraphVerticle * start,GraphVerticle * end) const
{
    return (edge(start,end)!=NULL);
}

bool Graph::addEdge(GraphVerticle * start,GraphVerticle * end, qreal p_weight)
{
    bool result;


    if(!edgeExists(start,end) && !edgeExists(end,start))
        {
        GraphEdge *edge =new GraphEdge(this,start,end,p_weight);
        m_edges.insert(index(start,end),edge);
        addItem(edge);
        result= true;
        }
    else
        {
        emit showExternInfo(trUtf8("Krawedz już istnieje..."));
        result= false;
        }

    remapAll();
    return result;
}
void Graph::removeEdge(GraphEdge * edge)
{

    delete edge;
    m_edges.remove(m_edges.key(edge));
    remapAll();
}
void Graph::removeEdge()
{
    changeState(RemovingEdge);
}

QList<GraphEdge *> Graph::incidental(const GraphVerticle * vert) const
{
    QList<GraphEdge *> result;
    foreach(GraphEdge * edge,m_edges)
    {
        if(edge->isIncidental(vert))
            result.append(edge);
    }
    return result;
}
QList<GraphVerticle *> Graph::incidental(const GraphEdge * edge) const
{
    QList<GraphVerticle *> result;
    result.append(edge->start());
    result.append(edge->end());
    return result;
}
QList<GraphVerticle *> Graph::neighbours(const GraphVerticle * vert) const
{
    QList<GraphVerticle *> result;

    foreach(GraphEdge * ed,m_edges)
    {
        if(ed->start()==vert)
            result.append(ed->end());
        if(ed->end()==vert)
            result.append(ed->start());
    }

    return result;
}
/*!
* Accesor function for all edges
* @return List of pointers to all edges present in graph
*/
QList<GraphEdge *> Graph::edges() const
{
    return m_edges.values();
}
/*!
* Accesor for all verticles in graph
* @return List of all pointers to verticles
*/
QList<GraphVerticle *> Graph::verticles() const
{
    return m_verticles;
}
void Graph::mousePressEvent( QGraphicsSceneMouseEvent * event )
{
    if(event->buttons() & Qt::LeftButton)
        {
        switch (m_state) {
            case AddingNewVerticle:
                {
                    if(event->buttons() | Qt::LeftButton)
                        {
                        addVerticle(event->scenePos());
                        emit showExternInfo(trUtf8("Dodano nowy wierzchołek"));
                        }
                }
                break;
            case AddingNewEdge:
                {
                    m_line = new QGraphicsLineItem(QLineF(event->scenePos(),
                            event->scenePos()));
                    addItem(m_line);
                    break;
                }
                break;
            case RemovingEdge:
                {
                    QList<QGraphicsItem *> tmp=items(event->scenePos());

                    while(!tmp.isEmpty())
                        {
                        if(tmp.first()->type()==GraphEdge::Type)
                            break;
                        else
                            tmp.removeFirst();
                        }
                    if(!tmp.isEmpty())
                        {
                        GraphEdge * edge=qgraphicsitem_cast<GraphEdge *>(tmp.first());
                        if(edge)
                            {
                            removeEdge(edge);
                            emit showExternInfo(trUtf8("Usunięto krawędź"));
                            }
                        }
                    else
                        {
                        event->ignore();
                        }
                }
                break;
            case RemovingVerticle:
                {
                    QList<QGraphicsItem *> tmp=items(event->scenePos());
                    while(!tmp.isEmpty())
                        {
                        if(tmp.first()->type()==GraphVerticle::Type)
                            break;
                        else
                            tmp.removeFirst();
                        }
                    if(!tmp.isEmpty())
                        {
                        GraphVerticle * vert=qgraphicsitem_cast<GraphVerticle *>(tmp.first());
                        if(vert)
                            {
                            removeVerticle(vert);
                            emit showExternInfo(trUtf8("Usunięto wierzchołek"));
                            }
                        }
                    else
                        {
                        event->ignore();
                        }
                }
                break;
            default:
                {
                    QList<QGraphicsItem *> tmp=items(event->scenePos());
                    while(!tmp.isEmpty())
                        {
                        if(tmp.first()->type()==GraphVerticle::Type)
                            break;
                        else
                            tmp.removeFirst();
                        }
                    if(!tmp.isEmpty())
                        {
                        GraphVerticle * vert=qgraphicsitem_cast<GraphVerticle *>(tmp.first());
                        if(vert)
                            {
                            emit verticleClicked(vert);
                            }
                        }
                    else
                        {
                        event->ignore();
                        }
                }
                break;
        }
        }
    else if(event->buttons() & Qt::RightButton)
        {
        emit showExternInfo(trUtf8("Powrocono do normalnego trybu"));
        changeState(IdleEdit);
        }
}
void Graph::mouseMoveEvent ( QGraphicsSceneMouseEvent * event )
{

    if(event->buttons() & Qt::LeftButton)
        {
        switch (m_state) {
            case AddingNewEdge:
                {
                    if (m_line !=NULL) {
                        QLineF newLine(m_line->line().p1(), event->scenePos());
                        m_line->setLine(newLine);
                    }
                }
                break;
            default:
                break;
        }
        }
}
void Graph::mouseReleaseEvent ( QGraphicsSceneMouseEvent * event)
{
    switch (m_state) {
        case AddingNewEdge:
            if (m_line !=NULL) {

                QList<QGraphicsItem *> startItems = items(m_line->line().p1());
                if (startItems.count() && startItems.first() == m_line)
                    startItems.removeFirst();
                QList<QGraphicsItem *> endItems = items(m_line->line().p2());
                if (endItems.count() && endItems.first() == m_line)
                    endItems.removeFirst();

                removeItem(m_line);
                delete m_line;

                while(!startItems.isEmpty() && !(startItems.first()->type() == GraphVerticle::Type))
                    {
                    startItems.removeFirst();
                    }
                while(!endItems.isEmpty() && !(endItems.first()->type() == GraphVerticle::Type))
                    {
                    endItems.removeFirst();
                    }

                if (startItems.count() > 0 && endItems.count() > 0 &&
                        startItems.first()->type() == GraphVerticle::Type &&
                        endItems.first()->type() == GraphVerticle::Type &&
                        startItems.first() != endItems.first())
                    {
                    GraphVerticle *startItem = qgraphicsitem_cast<GraphVerticle *>(startItems.first());
                    GraphVerticle *endItem = qgraphicsitem_cast<GraphVerticle *>(endItems.first());


                    qreal weight=1;
                    switch (m_weightScheme) {
                        case singleUnit:
                            weight=1;
                            break;
                        case geometricalDistance:
                            weight=QLineF(startItem->pos(),endItem->pos()).length();
                            break;
                        case manhattanDistance:
                            weight=(startItem->pos()-endItem->pos()).toPoint().manhattanLength();
                            break;
                        case Random1_10:
                            weight=(qrand()*1.0/RAND_MAX)*9+1;
                            break;
                        case Random1_100:
                            weight=(qrand()*1.0/RAND_MAX)*99+1;
                            break;
                        case Random0_10:
                            weight=(qrand()*1.0/RAND_MAX)*10;
                            break;
                        case Random0_100:
                            weight=(qrand()*1.0/RAND_MAX)*100;
                            break;
                    }


                    if(addEdge(startItem,endItem,weight))
                        {
                        emit showExternInfo(trUtf8("Dodano krawędź..."));
                        }
                    }
                m_line = 0;
            }
            break;
        default:
            break;
    }

}

/*!
* resets edges formatting
*/
void Graph::resetEdgesFormat()
{
    foreach(GraphEdge * edge, m_edges)
    {
        edge->resetFormat();
    }
}
/*!
* Reset verticles formatting
*/
void Graph::resetVerticlesFormat()
{
    foreach(GraphVerticle * vert, m_verticles)
    {
        vert->resetFormat();
    }
}
Graph::~Graph()
{
    qDeleteAll(m_verticles);
    qDeleteAll(m_edges);
    m_verticles.clear();
    m_edges.clear();
}

QDataStream & operator<< (QDataStream & stream,const Graph & graph)
{
    stream << graph.verticles().size();
    stream << graph.edges().size();
    foreach(GraphVerticle * vert, graph.verticles())
    {
        Q_ASSERT(vert->symbol().length()==1);
        stream << vert->symbol().at(0);
        stream << vert->pos();
    }
    foreach(GraphEdge * edge, graph.edges() )
    {
        stream << edge->start()->symbol().at(0);
        stream << edge->end()->symbol().at(0);
        stream << edge->weight();
    }
    return stream;
}
QDataStream & operator>> (QDataStream & stream,Graph&  graph)
{
    int edgeCount, vertCount;
    stream >> vertCount;
    stream >> edgeCount;

    qDeleteAll(graph.m_verticles);
    qDeleteAll(graph.m_edges);
    graph.m_verticles.clear();
    graph.m_edges.clear();

    QMap<QChar, GraphVerticle*> newVerticles;

    for (int i = 0; i < vertCount; ++i) {
        QChar symbol;
        QPointF where;
        stream >> symbol;
        stream >> where;
        newVerticles[symbol]=graph.addVerticle(where);
        newVerticles[symbol]->setSymbol(symbol);
    }
    for (int i = 0; i < edgeCount; ++i) {
        QChar start, end;
        qreal weight;
        stream >> start;
        stream >> end;
        stream >> weight;
        graph.addEdge(newVerticles[start],newVerticles[end],weight);
    }

    return stream;
}

