/*
 * Graph.h
 *
 *  Created on: 2009-06-07
 *      Author: Wesoly
 */

#ifndef GRAPH_H_
#define GRAPH_H_

#include <QGraphicsScene>
#include <QList>
#include <QMap>
#include <QPointF>

class GraphEdge;
class GraphVerticle;
class QGraphicsSceneMouseEvent;

/*!
 * Class representing graph. It has human interfacje for sombe basic info, and easy acces to internal data via
 * \sa    QList<GraphEdge *> edges() const;
 * \sa    QList<GraphVerticle *> verticles() const;
 */
class Graph : public QGraphicsScene
{
       Q_OBJECT
    friend QDataStream & operator<< (QDataStream & stream,const Graph & model);
    friend QDataStream & operator>> (QDataStream & stream,Graph&  model);
    friend class GraphEdge;
    friend class GraphVerticle;
    friend class AbstractSolver;

public:
    static const int circleRadius=10; //!< Initial verticle circle radius

    /*!
     * Adding edges weight scheme types
     */
    enum weightScheme
    {
        singleUnit,         //!< singleUnit
        geometricalDistance,//!< geometricalDistance
        manhattanDistance,  //!< manhattanDistance
        Random1_10,         //!< Random1_10
        Random1_100,        //!< Random1_100
        Random0_10,         //!< Random0_10
        Random0_100,        //!< Random0_100
    };
    /*!
     * Setn new weigh scheme for new edges
     * @param scheme new Scheme
     */
    void setNewEgdesWeightScheme(weightScheme scheme);

private:
    /*!
     * Represents graph state, useful while editing (solving uses Idle state only)
     */
    enum CurrentState{
        IdleEdit,         //!< Editing inactive, solving or just watching
        AddingNewVerticle,//!< Adding new verticle
        AddingNewEdge,    //!< Adding new edge
        RemovingEdge,     //!< Removing edge
        RemovingVerticle, //!< Removing verticle
    };
    /*!
     * represents index for edges map
     */
    typedef QPair<GraphVerticle *,GraphVerticle *> index;
    weightScheme m_weightScheme;
    QList<GraphVerticle *> m_verticles; //!< List of all verticles
    QMap<index,GraphEdge *> m_edges;    //!< Map of edges, with matrix-index based access
    CurrentState m_state;               //!< Graph state
    GraphVerticle * m_tmp;              //!< Temporary graph verticle used when adding edges
    QGraphicsLineItem * m_line;         //!< Temporary line, used when adding edges
    /*!
     * This function reasign letters for all Verticles, usefull after deletions
     * iit also updates (redraws) all connections
     */
    void remapAll();
signals:
    /*!
     * Graph emits this when it is needed to show extra info (warnings, instructions)
     * \param info Info to be shown
     */
    void showExternInfo(QString info);

    /*!
     * Graph emits this when it changes it's state
     * \param state New graph state
     */
    void stateChanged(QString state);

    /*!
     * Graph emits this when verticle is clicked
     * \param vert Pointer to clicked verticle
     */
    void verticleClicked(GraphVerticle* vert);

public slots: //Slots used for graph construction only
/*!
 * Used for graph creating only, binded to apropriate button
 */
    void addVerticle();
    /*!
     * Used for graph creating only, binded to apropriate button
     * \return Newly created verticle
     */
    GraphVerticle * addVerticle(QPointF where);
    /*!
     * Used for graph creating only, binded to apropriate button
     */
    void removeVerticle();
    /*!
     * Used for graph creating only, binded to apropriate button
     */
    void removeVerticle(GraphVerticle *);
    /*!
     * Used for graph creating only, binded to apropriate button
     */
    void addEdge();
    /*!
     * Used for graph creating only, binded to apropriate button
     */
    bool addEdge(GraphVerticle * start,GraphVerticle * end, qreal p_weight = std::numeric_limits<qreal>::infinity());
    /*!
     * Used for graph creating only, binded to apropriate button
     */
    void removeEdge();
    /*!
     * Used for graph creating only, binded to apropriate button
     */
    void removeEdge(GraphEdge * edge);


private:
    QList<GraphEdge *> incidental(const GraphVerticle * vert) const;
    QList<GraphVerticle *> incidental(const GraphEdge * edge) const;
    QList<GraphVerticle *> neighbours(const GraphVerticle * vert) const;
    void mousePressEvent ( QGraphicsSceneMouseEvent * event );
    void mouseMoveEvent ( QGraphicsSceneMouseEvent * event );
    void mouseReleaseEvent ( QGraphicsSceneMouseEvent * event );
    void changeState(CurrentState state);

public:
    /*!
     * Default constructor, created graph is empty
     */
    Graph();

    GraphEdge * edge(GraphVerticle * start,GraphVerticle * end);

    const GraphEdge * edge(GraphVerticle * start,GraphVerticle * end) const;

    bool edgeExists(GraphVerticle * start,GraphVerticle * end) const;

    QList<GraphEdge *> edges() const;

    QList<GraphVerticle *> verticles() const;

    void resetEdgesFormat();

    void resetVerticlesFormat();

    virtual ~Graph();
};

QDataStream & operator<< (QDataStream & stream,const Graph & graph);
QDataStream & operator>> (QDataStream & stream,Graph&  graph);

#endif /* GRAPH_H_ */
