/*
 * GraphEdge.h
 *
 *  Created on: 2009-06-07
 *      Author: Wesoly
 */

#ifndef GRAPHEDGE_H_
#define GRAPHEDGE_H_

#include <QGraphicsLineItem>
class Graph;
class GraphVerticle;

class GraphEdge : public QGraphicsLineItem
{
public:
    /*!
     * Simple enum used for creating item type
     */
    enum { Type = UserType + 1 };//!< Type of object, used for selecting it from scene
    /*!
     * Method for checking item type
     * @return Enumerated item type, named \sa Type
     */
    int type() const{return Type;}
private:
    GraphVerticle * const m_startVerticle; //!< Starting verticle, it is constant pointer to mutable value
    GraphVerticle * const m_endVerticle;   //!< Ending verticle, it is constant pointer to mutable value
    Graph * const m_graf;                  //!< Graph pointer, it is constant pointer to mutable value
    QGraphicsSimpleTextItem * m_weightItem;//!< Image containg edge weight
    qreal m_weight;                        //!< Edge weight
    /*!
     * Copy constructor, currently unused
     * @param pattern Orginal edge to be copied
     */
    GraphEdge(const GraphEdge &pattern);
public:
    /*!
     * Constructor. Params are needed becouse they are initialized ther (they're consts)
     * @param graf Graph that this edge belongs to
     * @param start Starting verticle
     * @param end Ending verticle.
     * @return
     */
    GraphEdge(Graph * graf,GraphVerticle * start,GraphVerticle * end, qreal p_weight = std::numeric_limits<qreal>::infinity());

    /*!
     * \property m_weight
     * @return Current edge weight
     */
    qreal weight() const;

    /*!
     * Redraws edge according to it's properties
     */
    void redraw();
    /*!
     * Changes edge width
     * @param width New pen width to be used
     */
    void setWidth(qreal width);

    /*!
     * Changes edge color
     * @param color New pen color to be aplied
     */
    void setColor(Qt::GlobalColor color);

    /*!
     * Changes edge drawing style
     * @param style New Pen style
     */
    void setStyle(Qt::PenStyle style);

    /*!
     * Checks if Edge is incidental with verticle
     * @param vert Verticle to be chekced
     * @return TRue if they're incidental, false otherwise
     */
    bool isIncidental(const GraphVerticle * vert) const
    {
        return (vert==m_endVerticle || vert==m_startVerticle);
    }

    /*!
     * Creates list of incidental erticles (start and end)
     * @return List of incidental verticles
     */
    QList<GraphVerticle*> incidental() const;

    /*!
     * Begining of this edge
     * @return Starting verticle
     */
    GraphVerticle * start() const {return m_startVerticle;};

    /*!
     * End of this edge pointer
     * @return Ending verticle pointer
     */
    GraphVerticle * end() const {return m_endVerticle;};

    /*!
     * Create edge name, which consist of starting and ending symbol, e.g CE
     * @return Created name
     */
    QString getName() const;

    /*!
     * Resets item formating (styles, colors)
     */
    void resetFormat();

    /*!
     * Destructor
     */
    virtual ~GraphEdge();
};

#endif /* GRAPHEDGE_H_ */
