#include "MintySolver.h"

#include <qDebug>
#include <QListWidgetItem>
#include <QTimer>
#include <QCloseEvent>

#include "Graph.h"
#include "GraphVerticle.h"
#include "GraphEdge.h"

MintySolver::MintySolver(Graph *graph, QWidget *parent)
    : AbstractSolver(graph, parent)
{
	QWidget * rightPanel = new QWidget;
	ui.setupUi(rightPanel);
	addRightPanel(rightPanel);

	algorithmSteps[0] = & MintySolver::findConnections;
	algorithmSteps[1] = & MintySolver::chooseMinValueConnection;
	algorithmSteps[2] = & MintySolver::checkStopCondition;
	algorithmSteps[3] = & MintySolver::end;

	ui.shortestWaysFrame->hide();
	resetGraph();

	disableStepForward();
	if(!checkIfConnected()) setInstruction(trUtf8("Wprowadzony graf jest niespójny!!!"));
	else
	{
		initializeLists();
		chooseStartVerticle();
	}
	this->setAlgorithmName(trUtf8("Algorytm Minty'ego"));
}

MintySolver::~MintySolver()
{

}
void MintySolver::stepForward()
{
	(this->*algorithmSteps[m_nextStep])();
}
void MintySolver::closeEvent(QCloseEvent * event)
{
	resetGraph();
	event->accept();
}

void MintySolver::initializeLists()
{
	m_verticles.clear();
	m_verticlesValues.clear();
	foreach(GraphVerticle * v, m_graph->verticles()){
		m_verticles.insert(v, true);
		m_verticlesValues.insert(v, 0);
	}
}
void MintySolver::resetGraph()
{
	m_graph->resetEdgesFormat();
	m_graph->resetVerticlesFormat();
}
bool MintySolver::checkIfConnected()
{
	foreach(GraphVerticle * v, m_graph->verticles()){
		m_verticles.insert(v, true);
	}
	ccDFS((m_graph->verticles()).at(0));

	foreach(GraphVerticle * v, m_graph->verticles()){
		if(m_verticles.value(v)) return false;
	}

	return true;
}

void MintySolver::chooseStartVerticle()
{
	setInstruction(trUtf8("Wybierz wierzchołek startowy."));
	connect(m_graph, SIGNAL(verticleClicked(GraphVerticle*)), this, SLOT(setStartVerticle(GraphVerticle*)));
}


void MintySolver::setStartVerticle(GraphVerticle * v)
{
	setInstruction("...");
	v->setRadius(15);
	addLogLine(trUtf8("Wybrano wierzchołek '") + v->symbol() + trUtf8("' jako startowy."));
	setAsClosed(v, 0);
	startVerticle = v;
	disconnect(m_graph, SIGNAL(verticleClicked(GraphVerticle*)), this, SLOT(setStartVerticle(GraphVerticle*)));
	m_nextStep = 0;
	enableStepForward();
}

void MintySolver::ccDFS(GraphVerticle * v)
{
	m_verticles[v] = false;
	QList<GraphVerticle *> vList;
	foreach(GraphVerticle * ve, v->neighbours()){
		if(m_verticles.value(ve)) vList.append(ve);
	}
	foreach(GraphVerticle * ve, vList){
		if(m_verticles.value(ve)) ccDFS(ve);
	}
}
void MintySolver::setAsClosed(GraphVerticle * vert, qreal value)
{
	vert->setColor(Qt::green);
	vert->setBorderWidth(3);

	m_verticles[vert] = false;
	m_verticlesValues[vert] = value;
	ui.closedVerticlesList->addItem((vert->symbol()) + "(" + QString::number(value) + ")");
	m_closedVerticles.append(vert);
	addLogLine(trUtf8("Dodano wierzchołek '") + vert->symbol() + trUtf8("' do listy wierzchołków odwiedzonych i przypisano mu wartość: ") + QString::number(value) + ".");
}
void MintySolver::setAsClosed(GraphEdge * ed)
{
	ed->setColor(Qt::green);
	ed->setWidth(2);
	m_closedEdges.append(ed);
}

void MintySolver::setAsActive(GraphEdge * ed)
{
	ed->setStyle(Qt::DashLine);
	addLogLine(trUtf8("Dodano krawędź '") + ed->getName() + trUtf8("' do listy krawędzi bieżących."));
}

qreal MintySolver::edgeSum(GraphEdge * ed)
{
	if(!m_verticles[ed->start()]) return (m_verticlesValues[ed->start()] + ed->weight());
	if(!m_verticles[ed->end()]) return (m_verticlesValues[ed->end()] + ed->weight());
	return 0;
}
QString MintySolver::shortestWay(GraphVerticle * v)
{
	QList<GraphVerticle *> verticles;
	verticles.append(v);
	GraphVerticle * last = v;

	for(; last->symbol() != startVerticle->symbol();)
	{
		foreach(GraphEdge * ed, last->incidental()){
			if(isClosed(ed)){
				if(m_closedVerticles.indexOf(secondVerticle(ed, last)) < m_closedVerticles.indexOf(last)){
					last = secondVerticle(ed, last);
					verticles.append(last);
				}
			}
		}
	}

	QString way(startVerticle->symbol());
	for (int i = verticles.count()-2; i>=0; --i) {
		way += "->" + verticles.at(i)->symbol();
	}
	return way;
}

bool MintySolver::isClosed(GraphEdge * ed)
{
	if(m_closedEdges.indexOf(ed) != -1) return true;
	return false;
}

GraphVerticle * MintySolver::secondVerticle(GraphEdge *ed, GraphVerticle *v)
{
	if(ed->start()->symbol() != v->symbol()) return ed->start();
	return ed->end();
}

void MintySolver::findConnections()
{
	setInstruction(trUtf8("Szukamy krawędzi, których jeden koniec jest oznaczony i dodajemy je do listy bieżących. Jednoczenie obliczamy sumę oznaczeń wierzchołka i jego krawędzi."));
	m_activeEdges.clear();
	foreach(GraphVerticle * v, m_graph->verticles())
	{
		if(!m_verticles[v])
		{
			foreach(GraphVerticle * neighbour, v->neighbours())
			{
				if(m_verticles[neighbour])
				{
					GraphEdge * ed = m_graph->edge(v, neighbour);
					m_activeEdges.append(ed);
					setAsActive(ed);
					ui.activeEdgesList->addItem(ed->getName() + " (" + QString::number(m_verticlesValues[v] + ed->weight()) + ")");
				}
			}
		}
	}
	m_nextStep = 1;
}
void MintySolver::chooseMinValueConnection()
{
	setInstruction(trUtf8("Z listy krawedzi bieżących wybieramy krawędź o najmniejszej sumie. Jej koniec oznaczamy wartością tejże sumy."));

	qreal min = edgeSum(m_activeEdges.first());
	GraphEdge * minEdge = m_activeEdges.first();
	foreach(GraphEdge * ed, m_activeEdges)
	{
		if(edgeSum(ed) < min)
		{
			min = edgeSum(ed);
			minEdge = ed;
		}
	}
	addLogLine(trUtf8("Jako krawędź o najmniejszej sumie wybrano krawędź '") + minEdge->getName() +"'.");
	setAsClosed(minEdge);
	if(m_verticles[minEdge->start()]) setAsClosed(minEdge->start(), edgeSum(minEdge));
	if(m_verticles[minEdge->end()]) setAsClosed(minEdge->end(), edgeSum(minEdge));
	m_nextStep=2;
}
void MintySolver::checkStopCondition()
{
	foreach(GraphEdge * ed, m_activeEdges)
	{
		ed->setStyle(Qt::SolidLine);
	}
	m_activeEdges.clear();
	ui.activeEdgesList->clear();
	addLogLine(trUtf8("Wyczyszczono listę krawędzi bieżących."));

	setInstruction(trUtf8("Sprawdzamy czy wszsytkie wierzchołki są oznaczone."));

	bool flag = false;
	foreach(GraphVerticle * v, m_graph->verticles())
	{
		if(m_verticles[v]) flag = true;
	}

	if(flag)
	{
		addLogLine(trUtf8("Nie wszsytkie wierzchołki zostały oznaczone. Kontynuujemy wykonywanie algorytmu."));
		m_nextStep = 0;
	}
	else m_nextStep = 3;
}

void MintySolver::end()
{
	foreach(GraphVerticle * v, m_graph->verticles()){
		if(v->symbol() != startVerticle->symbol()){
			ui.shortestWaysList->addItem(v->symbol() + "(" + QString::number(m_verticlesValues[v]) + ")" + ": " + shortestWay(v));
		}
	}
	ui.workFrame->hide();
	ui.shortestWaysFrame->show();

	setInstruction(trUtf8("Wyznaczono najkrótsze drogi do wszystkich wierzchołków."));
	addLogLine(trUtf8("Wszystkie wierzchołki zostały oznaczone. Koniec pracy algorytmu."));
	disableStepForward();

}
