#
# xglut.pl -- Convert glut.h for Glut 3.7.6 into an Ada Binding.
#
# Gene Ressler, Feb 2003
#
use strict;

use vars qw|*SF *BF $ID $IDD $ST %Symbol_Table @if_val_stack|;

%Symbol_Table = 
  (
   _WIN32 => 1,
   _MSC_VER=> 1200,
   _M_IX86 => 1,
   _DLL => 1,
   GLUT_API_VERSION => 4,
   GLUT_DISABLE_ATEXIT_HACK => 1,
  );

$ID = '[a-zA-Z_][a-zA-Z0-9_]*';
$IDD = '[a-zA-Z_][a-zA-Z0-9_.]*';
$ST = '(?:int|char|void|GLfloat|GLdouble|GLint|GLenum)';

sub trace {
  # print STDERR @_;
}

sub expr_val {
  my $expr = shift;
  1 while $expr =~ s/defined\s*\(($ID)\)/exists $Symbol_Table{$1} ? 1 : 0/e;
  1 while $expr =~ s/($ID)/$Symbol_Table{$1}/e;
  trace "eval_expr: $expr\n";
  eval $expr;
}

sub preprocess {

  my $input = shift;
  my @output;
  my $line_no = 0;

  for (@$input) {

    chomp;
    $line_no++;

    trace "stack: ", join(",", @if_val_stack), "\n";
    trace "Process '$_'\n";

    #if
    my @vals = /^\s*#\s*if\s+(.*)$/;
    if (@vals) {
      my ($expr) = @vals;
      my $val = expr_val($expr);
      unshift @if_val_stack, $val;
      trace "if $expr = $val\n";
      next;
    }

    #ifdef
    my @vals = /^\s*#\s*ifdef\s+($ID)/;
    if (@vals) {
      my ($id) = @vals;
      my $val = expr_val("defined($id)");
      unshift @if_val_stack, $val;
      trace "ifdef $id = $val\n";
      next;
    }

    #ifndef
    my @vals = /^\s*#\s*ifndef\s+($ID)/;
    if (@vals) {
      my ($id) = @vals;
      my $val = expr_val("!defined($id)");
      unshift @if_val_stack, $val;
      trace "ifndef $id = $val\n";
      next;
    }

    #else
    my @vals = /^\s*#\s*else/;
    if (@vals) {
      die unless @if_val_stack > 0;
      $if_val_stack[0] = 1 - $if_val_stack[0];
      trace "else\n";
      next;
    }

    #endif
    my @vals = /^\s*#\s*endif/;
    if (@vals) {
      die unless @if_val_stack > 0;
      shift @if_val_stack;
      trace "endif\n";
      next;
    }

    # If there's a false on the stack, line is omitted.
    next if scalar grep { !$_ } @if_val_stack;

    # The rest is processed.
    push @output, [$line_no, $_];

    # define
    my @vals = /^\s*#\s*define\s+($ID)\s*(.*)$/;
    if (@vals) {
      my ($id, $val) = @vals;
      $Symbol_Table{$id} = $val;
      trace "$id := $val\n";
      next;
    }
  }
  \@output;
}

use vars qw($c_to_ada_type);
$c_to_ada_type =
  {
   
   # These are hacked to match Mark Kilgard's
   'unsigned int' 	=> 'Interfaces.C.unsigned',
   'unsigned char' 	=> 'Interfaces.C.unsigned_char',
   'signed char'	=> 'Interfaces.C.signed_char',
   'short' 		=> 'Interfaces.C.short',
   'char'               => 'Interfaces.C.char',
   'int' 		=> 'Integer',
   'unsigned short' 	=> 'Interfaces.C.unsigned_short',
   'float' 		=> 'Interfaces.C.C_float',
   'double' 		=> 'Interfaces.C.double',
   'void' 		=> 'null record',
   'GLenum'             => 'GL.GLenum',
   'GLfloat'		=> 'GL.GLfloat',
   'GLdouble'		=> 'GL.GLdouble',
   'GLint'		=> 'GL.GLint',
  };

# Translate primitive types.
sub xp {
  my $primitive_type = shift;
  # print "xp: $primitive_type\n";
  return $c_to_ada_type->{$primitive_type} ||
    die "type '$primitive_type' is unknown";
}

# Do the real work of translating an argument type from C to Ada.
sub xt {
  my $type = shift;
  
  # Clobber white space.
  $type =~ s/^\s+//;
  $type =~ s/\s+$//;
  $type =~ s/\s+/ /g;
  
  my $base_type = $type;
  $base_type =~ s/ *\*//g;
  $base_type =~ s/^const\s*//;
  my $xlated_base_type = xp $base_type;
  
  return $xlated_base_type if $type =~ /^[a-zA-Z ]+$/;
  
  # Void pointer.
  return "System.Address" if $type =~ /^void\s*\*$/;

  # Pointer to char.
  return "Interfaces.C.Strings.Chars_Ptr"  if $type =~ /^(?:const\s+)?(?:unsigned\s+)?char\s*\*$/;

  # Pointer to pointer to char.
  return "access Interfaces.C.Strings.Chars_Ptr" if $type =~ /^(?:const\s+)?(?:unsigned\s+)?char\s*\*\*$/;
  
  # Simple pointer.
  return "access $xlated_base_type"  if $type =~ /^([a-z ]+)\s*\*$/;

  # pointer to constant.
  return "access constant $xlated_base_type" if $type =~ /^const\s+([a-z ]+)\s*\*$/;

  # Pointer to pointer.
  if ($type =~ /^([a-z ]+)\s*\*\s*\*$/) {

    # This isn't used, but I'll let it here to see how we 
    # could handle ** types if we needed.
    
    # Get type of a singly indirect pointer.
    my $ptr_type = xt("$1*");
    print STDERR "got $ptr_type on $1*\n";

    # Change package qualifiers to underscores.
    my $ada_name_for_xlated_base_type = $xlated_base_type;
    $ada_name_for_xlated_base_type =~ s/\./_/g;
  
    print SF "   type a_$ada_name_for_xlated_base_type is $ptr_type;\n";

    return "access a_$ada_name_for_xlated_base_type";
  }

  die "type '$type' is unknown";
}

# Translate an argument name from C to Ada.  This just deals with
# legal C var names that happen to be Ada reserved words.
use vars qw($created_id);
$created_id = 't00';

sub nt {
  my $name = shift;
  return ++$created_id unless $name;
  return "the_$name" if $name =~ /type|range|array|end|begin|string/i;
  return $name;
}

sub c2ada_args {

  my $cargs = shift;
  my $callback_type_prefix = shift;
  
  my $aargs = '';
  
  while ($cargs =~ /\S/) {

    my @vals;
    
    # Try callback type match
    @vals = $cargs =~ /^\s*void\s*\(GLUTCALLBACK\s*\*($ID)\s*\)\((.*?)\)/;
    if (@vals) {
      my ($id, $cfargs) = @vals;
      my $callback_type_name = $callback_type_prefix . '_Type';

      if ($cfargs =~ /^\s*void\s*$/) {
	print SF "\n   type $callback_type_name is access procedure;\n";
      }
      else {
	print SF "\n   type $callback_type_name is access procedure (\n";
	print SF &c2ada_args($cfargs, $callback_type_prefix);
	print SF "\n   );";
      }
      
      $aargs .= ";\n" if $aargs;
      $aargs .= ' 'x6;
      $aargs .= nt $id;
      $aargs .= " : ";
      $aargs .= $callback_type_name;
      
      # print "  id=$id cfargs=$cfargs\n";
      $cargs = $';
      $cargs =~ s/^\s*,\s*//;
      next;
    }

    # Try simple type match
    @vals = $cargs =~ /^\s*(const\s+)?((?:unsigned\s+)?$ST)\b\s*(\**)($ID)?/;
    if (@vals) {
      my ($const, $simple_type, $stars, $id) = @vals;

      $aargs .= ";\n" if $aargs;
      $aargs .= ' 'x6;
      $aargs .= nt $id;
      $aargs .= " : ";
      $aargs .= xt "$const $simple_type $stars";
      
      # print "  const=$const, st=$simple_type, *=$stars, id=$id\n";
      $cargs = $';
      $cargs =~ s/^\s*,\s*//;
      next;
    }
 
    die "can't parse type $cargs";
  }

  $aargs;
}

# To the translation.
sub emit_spec {

  my $input = shift;
  my $header_comment = shift;

  # Commonly used regex for a C identifier.
  my $ID = '[A-Za-z_0-9]+';

  # Emit boilerplate.
  print SF $header_comment;
  print SF "\n";

  print SF "with Win32.GL;\n";
  print SF "with Interfaces.C;\n";
  print SF "with Interfaces.C.Strings;\n";
  print SF "with System.Storage_Elements;\n\n";
  
  print SF "package Win32.Glut is\n\n";
  print SF "   package GL renames Win32.GL;\n\n";

  print BF $header_comment;
  print BF "\npackage body Win32.Glut is\n\n";
  print BF "   pragma Linker_Options (\"-lopengl32\");\n";
  print BF "   pragma Linker_Options (\"-lglu32\");\n";
  print BF "   pragma Linker_Options (\"-lglut32\");\n";

  my $in_comment_p;
  
 PAIR: foreach my $line_pair (@$input) {

    my $line_no = $line_pair->[0];
    $_ = $line_pair->[1];

    chomp;

  LINE: {
	    
      # in multiline comment; look for end
      if ($in_comment_p) {
	if (m#\*/#) {
	  print SF "   -- $`\n";
	  $_ = $';
	  $in_comment_p = 0;
	  redo LINE;
	} else {
	  print SF "   -- $_\n";
	  next PAIR;
	}
      }
    
      # in-line comment
      my @vals = m#^\s*/\*(.*?)\*/#;
      if (@vals) {
	my ($cmt) = @vals;
	print SF "   -- $cmt\n";
	$_ = $';	       # Line is now everything after comment.
	redo LINE;
      }
    
      # multiline comment
      my @vals = m#^\s*/\*(.*)#;
      if (@vals) {
	my ($cmt) = @vals;
	print SF "   -- $cmt\n";
	$in_comment_p = 1;
	next PAIR;
      }
    
      # vertical whitespace
      unless (/\S/) {
	print SF "\n";
	next PAIR;
      }

      # Font flags of form #define GLUT_STROKE_ROMAN ((void*)0)
      my @vals = /^#define\s+($ID)\s+\(\(void\*\)($ID)\)/o;
      if (@vals) {
	my ($id, $dec_val) = @vals;
	$id .= ' ' x (36 - length($id));
	print SF "   $id : constant  System.Address\n";
	print SF "     := System.Storage_elements.To_Address($dec_val); --  glut.h:$line_no\n";
	next PAIR;
      }

      # Decimal and symbolic defines, possibly with enum type cast.
      my @vals = /^#define\s+($ID)\s+($ID)/o; 
      @vals = /^#define\s+($ID)\s+\(\($ID\)\s*($ID)\)/o unless @vals;
      if (@vals) {
	my ($id, $dec_val) = @vals;
	$id .= ' ' x (36 - length($id));
	print SF "   $id : constant := $dec_val; --  glut.h:$line_no\n";
	next PAIR;
      }

      # void functions.
      my @vals = /^\s*GLUTAPI\s+void\s+APIENTRY\s+($ID)\((.*)\)/;
      if (@vals) {
	my ($proc_id, $cargs) = @vals;
	# print SF "\nargs=$cargs\n";
	if ($cargs =~ /^\s*void\s*$/) {
	  print SF "\n   procedure $proc_id; --  gl.h:$line_no\n";
	  print SF qq|   pragma Import (Stdcall, $proc_id, "$proc_id");\n|;
	} else {

	  my $ada = "\n   procedure $proc_id(\n";
	  $ada .= c2ada_args($cargs, $proc_id);
	  $ada .= "\n   ); --  glut.h:$line_no\n";

	  print SF $ada;
	  print SF qq|   pragma Import (Stdcall, $proc_id, "$proc_id");\n|;

	  # print Ada string interface variant if necessary
	  if ($ada =~ s/:\s*Interfaces.C.Strings.Chars_Ptr\b/: String/g) {

	    print SF $ada;

	    # Build the alternate interface procedure code.
	    # Knock out line breaks, semi, and comment.
	    $ada =~ s/; --.*//g;

	    # Find string args.
	    my @str_args = $ada =~ /($ID)\s*:\s*String/iog;

	    # Build body
	    my $body = "   is\n";
	    foreach my $str_arg (@str_args) {
	      $body .= "     C_$str_arg : Interfaces.C.Strings.Chars_Ptr\n";
	      $body .= "        := Interfaces.C.Strings.New_String($str_arg);\n";
	    }
	    $body .= "   begin\n";

	    # Call is modified version of header.
	    my $call = $ada;

	    # Remove 'procedure'
	    $call =~ s/procedure //;

	    # Replace ada string args with C versions.
	    $call =~ s/($ID)\s*:\s*String/C_$1 : String/iog;

	    # Remove types
	    $call =~ s/\s*:\s*$IDD//go;

	    # Commas, not semis.
	    $call =~ s/;/, /g;

	    # Line breaks
	    $call =~ s/\n\s*//g;

	    # Finish up the body now.
	    $body .= "     $call;\n";
	    foreach my $str_arg (@str_args) {
	      $body .= "     Interfaces.C.Strings.Free(C_$str_arg);\n";
	    }
	    $body .= "   end $proc_id;\n";
	    print BF $ada . $body;
	  }
	}
	next PAIR;
      }

      # other than void functions.
      my @vals = /^\s*GLUTAPI\s+($ST)\s+APIENTRY\s+($ID)\((.*)\)/;
      if (@vals) {
	my ($crtn_type, $proc_id, $cargs) = @vals;
	# print SF "\nargs=$cargs\n";
	if ($cargs =~ /^\s*void\s*$/) {
 	  print SF "   function $proc_id return " . xt($crtn_type) . "; --  gl.h:$line_no\n";
	  print SF qq|   pragma Import (Stdcall, $proc_id, "$proc_id");\n|;
	} else {

	  my $artn_type = xt($crtn_type);
	  my $ada = "\n   function $proc_id(\n";
	  $ada .= c2ada_args($cargs, $proc_id);
	  $ada .= "\n   ) return $artn_type; --  glut.h:$line_no\n";

	  print SF $ada;
	  print SF qq|   pragma Import (Stdcall, $proc_id, "$proc_id");\n|;

	  # print Ada string interface variant if necessary
	  if ($ada =~ s/:\s*Interfaces.C.Strings.Chars_Ptr\b/: String/g) {

            # Print the modified interface.
	    print SF $ada;

	    # Build the alternate interface procedure code.
	    # Knock out line breaks, semi, and comment.
	    $ada =~ s/; --.*//g;

	    # Find string args.
	    my @str_args = $ada =~ /($ID)\s*:\s*String/iog;

	    # Build body
	    my $body = "   is\n";
	    $body .=   "     Rtn : $artn_type;\n";
	    foreach my $str_arg (@str_args) {
	      $body .= "     C_$str_arg : Interfaces.C.Strings.Chars_Ptr\n";
	      $body .= "        := Interfaces.C.Strings.New_String($str_arg);\n";
	    }
	    $body .= "   begin\n";

	    # Call is modified version of header.
	    my $call = $ada;

	    # Remove 'function'
	    $call =~ s/function //i;

	    # Replace ada string args with C versions.
	    $call =~ s/($ID)\s*:\s*String/C_$1 : String/iog;

	    # Remove types
	    $call =~ s/\s*:\s*$IDD//go;

	    # Commas, not semis.
	    $call =~ s/;/, /g;

	    # Line breaks
	    $call =~ s/\n\s*//g;

	    # Return clause;
	    $call =~ s/\).*/\)/;

	    # Add assignment.
	    $call = "Rtn := $call";

	    # Finish up the body now.
	    $body .= "     $call;\n";
	    foreach my $str_arg (@str_args) {
	      $body .= "     Interfaces.C.Strings.Free(C_$str_arg);\n";
	    }
	    $body .= "     return Rtn;\n";
	    $body .= "   end $proc_id;\n";
	    print BF $ada . $body;
	  }
	}
	next PAIR;
      }

      # Make everything else a comment if we're testing.
      # To see what our cases above are missing.
      # print SF "-- Junk: $_\n" if /\S/;
    }
  }

  print SF "end Win32.Glut;\n";
  print BF "end Win32.Glut;\n";
}

sub run {
  local *F;
  open(F, 'D:/Packages/glut-3.7.6src/include/gl/glut.h') || die;
  my @input = <F>;
  close(F);
  my $preprocessed_input = preprocess \@input;

  emit_spec $preprocessed_input, 

    q|
-- Copyright (c) Mark J. Kilgard, 1994, 1995, 1996, 1998.

-- This program is freely distributable without licensing fees  and is
-- provided without guarantee or warrantee expressed or  implied. This
-- program is -not- in the public domain.

--  modified for GNAT 3.11p/NT & Glut 3.7
--  Jerry van Dijk, 22-feb-99, jdijk@acm.org
--  use with accompanying DLL only
--
--  modified again for GNAT 3.14/NT & Glut 3.76
--  Gene Ressler, 22-jan-03, ressler@usma.edu
--  use with accompanying DLL only
|;

  #print join "\n", map { sprintf "%5d: %s", @$_ } @$preprocessed_input;
}

open(SF, ">win32-glut.ads") || die;
open(BF, ">win32-glut.adb") || die;

run;
