-- A version of the game Asteroids in Ada using OpenGL.
-- COL Gene Ressler.
--
-- This specification provides for communication between the graphics
-- state and the simulation state of the system.

with Win32.Gl;
use  Win32.Gl;

package Simulation_State is

   -- Size of the universe where simulation takes place.  It is a
   -- closed 2d universe.  Right and left edges are connected, as are
   -- top and bottom.
   Universe_X_Circumference : constant GlFloat := 1000.0;
   Universe_Y_Circumference : constant GlFloat := 700.0;

   -- Fire the ship's main engine.
   procedure Fire_Main_Engine(Is_Firing : in Boolean);

   -- Possible commands to the thrusters.
   type Firing_State_Type is
     (Spin_Left, Cancel_Spin_Left,
      Spin_Right, Cancel_Spin_Right);

   -- Fire the ship's thrusters.
   procedure Fire_Thrusters(Firing_State : Firing_State_Type);

   -- Fire a bullet from the ship.
   procedure Fire_Bullet;

   -- Advance the simulation clock forward.
   procedure Advance_Clock(New_Clock : in GlFloat);

   -- Return relevant facts about the space ship.
   procedure Get_Space_Ship_Geometry(Alive : out Boolean;
                                     X, Y, Theta : out GlFloat;
                                     Main_Engine_Firing : out Boolean);


   -- Distance between the origin of the space ship and the gun where
   -- bullets are fired.  So the coordinates of the gun are
   --   (X + Space_Ship_Gun_X * Cos(Theta), Y + Space_Ship_Gun_X * Sin(Theta))
   Space_Ship_Gun_X : constant GlFloat := 20.0;

   -- Return relevant facts about a bullet. This is an iterator.
   -- Called with Index=0, it will advance index to the index for the
   -- first bullet and return its geometry.  Each successive call
   -- automatically advances Index.  When there are no more bullets,
   -- it returns Index of 0.  So typical code for getting all the
   -- bullet information is:
   --
   -- Index := 0;
   -- loop
   --    Get_Bullet_Geometry(X, Y, Index);
   --    exit when Index = 0;
   --    ... process the x-y data here
   -- end loop;
   procedure Get_Bullet_Geometry(X, Y : out GlFloat;
                                 Index : in out Natural);

   -- Same as above for bullets, but for rocks instead.
   procedure Get_Rock_Geometry(X, Y, Radius : out GlFloat;
                               Index : in out Natural);

   -- Explosion particles are given merely by their x-y coordinates.
   type Particle_Geometry_Type is
      record
         X, Y : GlFloat;
      end record;

   -- List of particles represents an explosion.
   type Particle_Geometry_List_Type is Array (1..8) of Particle_Geometry_Type;

   -- Either the ship is exploding or a rock.
   type Object_Type is (Space_Ship_Object, Rock_Object);

   -- Particles and type wrapped together make an explosion geometry.
   type Explosion_Geometry_Type is
     record
        Particles : Particle_Geometry_List_Type;
        Object : Object_Type;
     end record;

   -- Gets successive explosion geometries using same iterator scheme
   -- as above for rocks and bullets.
   procedure Get_Explosion_Geometry(Explosion_Geometry : out Explosion_Geometry_Type;
                                    Index : in out natural);

   -- Get the current score.  Not exactly physics, but still part of the simulation.
   procedure Get_Score(Score, High_Score : out Natural);

   -- Reset the simulation state to initial conditions.
   procedure Reset;

end Simulation_State;
