--                              -*- Mode: Ada -*-
-- Filename        : pixmap.ads
-- Description     : Pixmaps
-- Author          : COL Ressler
-- Created On      : Sun Mar 10 14:59:06 2002
-- Last Modified By: COL Ressler
-- Last Modified On: .
-- Update Count    : 0
-- Status          : Unknown, Use with caution!

-- Use Ada automagic adjustment at assignment time and
-- finalization to free pixel storage.
with Ada.Finalization;
with Win32.GL; use Win32.GL;

package Pixmaps is

   -- Tell Ada we really want 8 bit bytes to represent colors.
   type Byte_Type is mod 2**8;
   for Byte_Type'Size use 8;

   -- A pixel is an RGB triple of bytes.
   type RGB_Pixel_Type is record
      Red, Green, Blue : Byte_Type;
   end record;

   for RGB_Pixel_Type'Alignment use 1;
   for RGB_Pixel_Type'Size use 24;

   -- A pixmap of RGB pixels.
   type Pixmap_Type is new Ada.Finalization.Controlled with private;

   -- Clear out the image from an existing pixmap.
   procedure Clear_Pixmap(Pixmap : in out Pixmap_Type);

   -- Exception raised if a file passed to the BMP
   -- file reader does not meet the BMP file spec.
   BMP_Format_Error : exception;

   -- Read a BMP file into the given pixmap.  Old
   -- contents of the pixmap are destroyed.
   procedure Read_BMP_File(File_Name : in String;
                           Pixmap : out Pixmap_Type);

   -- Write a BMP file using data from the given pixmal.
   procedure Write_BMP_File(File_Name : in String;
                            Pixmap : in Pixmap_Type);

   -- Use the bitmap to set up an openGL texture.
   -- The Identifier can be any unique integer value
   -- for each texture set up.
   procedure Set_OpenGL_Texture(Pixmap : in Pixmap_Type;
                                Identifier : in Integer);

   -- Retrieve pixmap sizes.
   function Height(Pixmap : in Pixmap_Type) return Natural;

   function Width(Pixmap : in Pixmap_Type) return Natural;

   procedure Get_Size(Pixmap : in Pixmap_Type;
                      Width, Height : out Natural);

   -- Return a pixel value.
   function Pixel(Pixmap : in Pixmap_Type;
                  Row, Column : in Natural) return RGB_Pixel_Type;
   pragma Inline(Pixel);

   procedure Get_Pixel(Pixmap : in Pixmap_Type;
                       Row, Column : in Natural;
                       Pixel : out RGB_Pixel_Type);
   pragma Inline(Get_Pixel);

   -- Set a pixel value.
   procedure Set_Pixel(Pixmap : in out Pixmap_Type;
                       Row, Column : in Natural;
                       Pixel : in RGB_Pixel_Type);

   -- XY versions of get and set are for convenience
   -- of those using (x,y) coordinates rather than (row, column).
   -- These functions just call the above with coordinates
   -- reversed.
   function Pixel_XY(Pixmap : in Pixmap_Type;
                     X, Y : in Natural) return RGB_Pixel_Type;

   pragma Inline(Pixel_XY);
   procedure Get_Pixel_XY(Pixmap : in Pixmap_Type;
                          X, Y : in Natural;
                          Pixel : out RGB_Pixel_Type);
   pragma Inline(Get_Pixel_XY);

private

   -- Store the pixels in a rectangular array.  By default GNAT won't pack this array,
   -- so each pixel will take 4 bytes.  Use pragma packed to get a more memory efficient
   -- but overall slower implementation.
   type RGB_Pixel_Array_Type is array (Natural range <>, Natural range <>) of RGB_Pixel_Type;
   pragma Pack(RGB_Pixel_Array_Type);

   type RGB_Pixel_Array_Ptr_Type is access RGB_Pixel_Array_Type;

   type Pixmap_Type is new Ada.Finalization.Controlled with
      record
         N_Rows, N_Columns : Natural := 0;
         Pixels : RGB_Pixel_Array_Ptr_Type;
      end record;

   -- Managers to keep trackof pixel array when pixmap is assigned or
   -- deleted (going out of scope or deallocated).
   procedure Adjust(Pixmap: in out Pixmap_Type);
   procedure Finalize(Pixmap : in out Pixmap_Type);

end Pixmaps;
