--                              -*- Mode: Ada -*-
-- Filename        : texturemap.adb
-- Description     : Sample of drawing objects with multiple texture maps.
-- Author          : COL Ressler
-- Created On      : Tue March 5, 2002
-- Last Modified By: .
-- Last Modified On: .
-- Update Count    : 0
-- Status          : Unknown, Use with caution!  Really!

-- IO for debugging.
with Ada.Text_IO; use Ada.Text_IO;
with Ada.Float_Text_IO; use Ada.Float_Text_IO;

-- Use the Windows binding to OpenGL.
-- We add a "use" clause because all the names in this
-- package start with "GL" anyway.
with Win32.GL; use Win32.GL;

-- Use the Windows binding to OpenGL Utilities package.
-- We add a "use" clause because all the names in this
-- package start with "Glut" anyway.
with Win32.Glu; use Win32.Glu;
with Win32.Glut; use Win32.Glut;

-- This gives us access to the command line parameters
-- provided when the user started the program.
with Ada.Command_Line;

-- This lets us pass strings to the OpenGL and GLUT calls
-- that expect them.  The binding is so thin, that we
-- have to use C-style strings rather than Ada strings.
-- C strings have a different internal representation.
with Interfaces.C.Strings;
use type Interfaces.C.unsigned;
use type Interfaces.C.unsigned_char;

-- This gives us access to very commonly used arithmetic like
-- sqrt(), sin(), and cos(), pi.
with Ada.Numerics.Generic_Elementary_Functions;
use Ada.Numerics;

-- Use our pixmap package.
with Pixmaps;
use Pixmaps;

procedure Texturemap is

   -- Constant string to use for window title.
   App_Title : constant String := "Rotating Textured Cube";

   -- Windows are represented by integers.
   Window : Integer := 0;

   -- Declare a new elementary functions package that lets
   -- us manipulate OpenGL's floating point numbers.
   package Numerics is new
     Ada.Numerics.Generic_Elementary_Functions (GlDouble);
   use Numerics;

   -- These are global so we can see them in Display, Idle,
   -- and Mouse Move callbacks.
   Last_Redraw_Time : Integer := 0;
   X_Speed, Y_Speed, X_Angle, Y_Angle : GlDouble := 0.0;

   -- Set up an array of pixmaps, one per face of the cube to be drawn.
   type Texture_Array_Type is array(2001..2006) of Pixmap_Type;
   Textures : Texture_Array_Type;

   -- This procedure is called by GLUT each time the window needs to be redrawn.
   -- Examples of when this might occur:
   --   o Window is initially created.
   --   o Window is restored after being minimized.
   --   o Window is brought to front or uncovered after being covered by other windows.
   --   o Window is resized.
   --   o Some other part of the program calls GlutPostRedisplay, which forces a redraw.
   procedure Display is
   begin

      -- See pg. 454 in F.S. Hill.
      GlClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);
      GlTexEnvI(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

      -- In this program, the camera stays still and we rotate the object.
      -- The initialization procedure left us in matrix mode MODEL_VIEW.
      GlPushMatrix;
      GlRotated(X_Angle, 1.0, 0.0, 0.0);
      GlRotated(Y_Angle, 0.0, 1.0, 0.0);

      -- Front
      GlBindTexture(GL_TEXTURE_2D, GLuint(Textures'First));
      GlBegin(GL_QUADS);
      GlTexCoord2f(0.0, 0.0); GlVertex3f(-1.0, -1.0,  1.0);
      GlTexCoord2f(2.0, 0.0); GlVertex3f( 1.0, -1.0,  1.0);
      GlTexCoord2f(2.0, 2.0); GlVertex3f( 1.0,  1.0,  1.0);
      GlTexCoord2f(0.0, 2.0); GlVertex3f(-1.0,  1.0,  1.0);
      GlEnd;

      -- Right
      GlBindTexture(GL_TEXTURE_2D, GLuint(Texture_Array_Type'First + 1));
      GlBegin(GL_QUADS);
      GlTexCoord2f(0.0, 0.0); GlVertex3f(1.0, -1.0,  1.0);
      GlTexCoord2f(2.0, 0.0); GlVertex3f(1.0, -1.0, -1.0);
      GlTexCoord2f(2.0, 2.0); GlVertex3f(1.0,  1.0, -1.0);
      GlTexCoord2f(0.0, 2.0); GlVertex3f(1.0,  1.0,  1.0);
      GlEnd;

      -- Back
      GlBindTexture(GL_TEXTURE_2D, GLuint(Texture_Array_Type'First + 2));
      GlBegin(GL_QUADS);
      GlTexCoord2f(0.0, 0.0); GlVertex3f( 1.0, -1.0, -1.0);
      GlTexCoord2f(2.0, 0.0); GlVertex3f(-1.0, -1.0, -1.0);
      GlTexCoord2f(2.0, 2.0); GlVertex3f(-1.0,  1.0, -1.0);
      GlTexCoord2f(0.0, 2.0); GlVertex3f( 1.0,  1.0, -1.0);
      GlEnd;

      -- Left
      GlBindTexture(GL_TEXTURE_2D, GLuint(Texture_Array_Type'First + 3));
      GlBegin(GL_QUADS);
      GlTexCoord2f(0.0, 0.0); GlVertex3f(-1.0, -1.0, -1.0);
      GlTexCoord2f(2.0, 0.0); GlVertex3f(-1.0, -1.0,  1.0);
      GlTexCoord2f(2.0, 2.0); GlVertex3f(-1.0,  1.0,  1.0);
      GlTexCoord2f(0.0, 2.0); GlVertex3f(-1.0,  1.0, -1.0);
      GlEnd;

      -- Top
      GlBindTexture(GL_TEXTURE_2D, GLuint(Texture_Array_Type'First + 4));
      GlBegin(GL_QUADS);
      GlTexCoord2f(0.0, 0.0); GlVertex3f(-1.0, 1.0,  1.0);
      GlTexCoord2f(2.0, 0.0); GlVertex3f( 1.0, 1.0,  1.0);
      GlTexCoord2f(2.0, 2.0); GlVertex3f( 1.0, 1.0, -1.0);
      GlTexCoord2f(0.0, 2.0); GlVertex3f(-1.0, 1.0, -1.0);
      GlEnd;

      -- Bottom
      GlBindTexture(GL_TEXTURE_2D, GLuint(Texture_Array_Type'First + 5));
      GlBegin(GL_QUADS);
      GlTexCoord2f(0.0, 0.0); GlVertex3f(-1.0, -1.0, -1.0);
      GlTexCoord2f(2.0, 0.0); GlVertex3f( 1.0, -1.0, -1.0);
      GlTexCoord2f(2.0, 2.0); GlVertex3f( 1.0, -1.0,  1.0);
      GlTexCoord2f(0.0, 2.0); GlVertex3f(-1.0, -1.0,  1.0);
      GlEnd;

      -- Send everything to the frame buffer.
      GlFlush;

      -- Restore MODEL_VIEW matrix.
      GlPopMatrix;

      -- Show what we just drew.
      GlutSwapBuffers;
   end;

   -- Idle procedure handles updating image for animation purposes.
   procedure Idle is
      Time : Integer := GlutGet (GLUT_ELAPSED_TIME);
      Elapsed : GlDouble := GlDouble(Time - Last_Redraw_Time);
   begin

      -- Redraw no more often than every few ms.
      if Elapsed > 20.0 then

         -- Update the view angles based on rotation speeds.
         X_Angle := X_Angle + X_Speed;
         Y_Angle := Y_Angle + Y_Speed;

         GlutPostWindowRedisplay(Window);

         -- Remember this is the last time we did a redraw.
         Last_Redraw_Time := Time;
      else
         -- Delay keeps us from hogging the CPU.
         delay 0.01;
      end if;
   end;

   -- Keep track of coordinate where mouse button went down.
   Mouse_Down : Boolean := False;
   X_Mouse_Down, Y_Mouse_Down : Integer;

   -- This procedure is called by GLUT each time the user presses or releases a mouse button.
   -- The button that was clicked and whether it is up or down is given by the
   -- parameters Button and State respectively.  Compare them to the given constants.
   procedure Mouse_Click(Button : Integer; -- GLUT_LEFT_BUTTON or GLUT_RIGHT_BUTTON
                         State : Integer;  -- GLUT_DOWN or GLUT_UP
                         X : Integer; Y : Integer) is
   begin
      if Button = GLUT_LEFT_BUTTON then
         if State = GLUT_DOWN then
            -- Remember where the press occurred.
            X_Mouse_Down := X;
            Y_Mouse_Down := Y;
            Mouse_Down := True;

            -- Freeze at mouse click.  This is GUI implementation decision.
            X_Speed := 0.0;
            Y_Speed := 0.0;

         else -- GLUT_UP!  Mouse was just released.
            Mouse_Down := False;
         end if;
      end if;
   end;

   -- This procedure is called frequently as the mouse is moved around the screen with
   -- a button down.  We use it to track the other end of the vector.
   procedure Mouse_Motion(X : Integer; Y : Integer) is
      -- Some intermediate values we need to calculate.
      X_Mouse_Vector, Y_Mouse_Vector : Integer;

      -- Experimentally determined constant to make the interface respond not
      -- to much and not too little to user input.
      Usability_Fudge_Factor : constant GlDouble := 0.005;
   begin
      if Mouse_Down then
         X_Mouse_Vector := X - X_Mouse_Down;
         Y_Mouse_Vector := Y - Y_Mouse_Down;

         -- Reset the speeds based on where mouse has moved since button down.
         Y_Speed := Usability_Fudge_Factor * GlDouble(X_Mouse_Vector);
         X_Speed := Usability_Fudge_Factor * GlDouble(Y_Mouse_Vector);
      end if;
   end;

   -- This procedure sets up OpenGL.  You will have to modify it for
   -- the problem at hand.
   procedure Setup_OpenGL(Title : in String; Window : out Integer) is
      Argc : aliased Integer;
      pragma Import (C, Argc, "gnat_argc");

      type Chars_Ptr_Ptr_Type is access Interfaces.C.Strings.chars_ptr;
      Argv : Chars_Ptr_Ptr_Type;
      pragma Import (C, Argv, "gnat_argv");

   begin
      GlutInit (Argc'Access, Argv);
      GlutInitDisplayMode (GLUT_DOUBLE or GLUT_RGB or GLUT_DEPTH);

      GlutInitWindowSize(640, 480);
      Window := GlutCreateWindow(Title);

      GlutDisplayFunc (Display'Unrestricted_Access);
      GlutIdleFunc (Idle'Unrestricted_Access);
      GlutMouseFunc(Mouse_Click'Unrestricted_Access);
      GlutMotionFunc (Mouse_Motion'Unrestricted_Access);

      GlEnable(GL_DEPTH_TEST);             -- Turn on hidden surface removal with z-buffer algorithm
      GlEnable(GL_TEXTURE_2D);             -- Turn on texture mapping.
      GlClearColor(1.0, 1.0, 1.0, 1.0);    -- Background white.

      -- Set up texture pixmaps by reading in six BMP files.
      declare
         Face : Character := '1';
      begin
         for Texture_Index in Textures'Range loop
            Read_BMP_File("Texture" & Face & ".bmp", Textures(Texture_Index));
            Set_OpenGL_Texture(Textures(Texture_Index), Texture_Index);
            Put_Line("Read pixmap for face " & Face & "...");
            Face := Character'Succ(Face);
         end loop;
      end;

      -- Set up stationary camera.
      GlViewPort(0, 0, 640, 480);
      GlMatrixMode(GL_PROJECTION);
      GlLoadIdentity;
      GluPerspective(60.0, 640.0/480.0, 1.0, 30.0);
      GlMatrixMode(GL_MODELVIEW);
      GlLoadIdentity;
      GlTranslated(0.0, 0.0, -4.0);
   end;

begin
   -- Get OpenGL ready to go and open a window.
   Setup_OpenGL(App_Title, Window);

   -- Tell Glut to handle events and call our callback functions
   -- at the correct time.
   GlutMainLoop;
end;
