# Convert Microsoft gl.h to an Ada95 binding.
# Gene Ressler (c) 2002

require 5.6.0;
use strict;

# Translate primitive types.
sub xp {
  my $primitive_type = shift;
  my $xlated_type = 
  {
      # These work fine if you "with interfaces.C".
      #		'unsigned int' 		=> 'Interfaces.C.unsigned',
      #		'unsigned char' 	=> 'Interfaces.C.unsigned_char',
      #		'signed char' 		=> 'Interfaces.C.signed_char',
      #		'short' 		=> 'Interfaces.C.short',
      #		'int' 			=> 'Interfaces.C.int',
      #		'unsigned short' 	=> 'Interfaces.C.unsigned_short',
      #		'float' 		=> 'Interfaces.C.C_float',
      #		'double' 		=> 'Interfaces.C.double',
      #		'void' 			=> 'null record',
      
      # These are equivalent and consistent with win32.
      'unsigned int' 	=> 'Win32.UINT',
      'unsigned char' 	=> 'Win32.UCHAR',
      'signed char' 	=> 'Win32.CHAR',
      'short' 		=> 'Win32.SHORT',
      'int' 		=> 'Win32.INT',
      'unsigned short' 	=> 'Win32.USHORT',
      'float' 		=> 'Win32.FLOAT',
      'double' 		=> 'Win32.DOUBLE',
      'void' 		=> 'Win32.VOID',

  } -> {$primitive_type};
  return $xlated_type if $xlated_type;
  die "type '$primitive_type' is unknown";
}

# Emit a pointer type just once when it is first encountered.
# Do nothing on subsequent occasions.
use vars qw(%ptr_types);
sub emit_ptr_type {
  local $_ = shift;
  return if exists $ptr_types{$_};
  $ptr_types{$_}++;
  if (/^a_GLvoid/) { 
    #		print "   subtype a_GLvoid is System.Address;\n"; 
    print "   subtype a_GLvoid is Win32.PVOID;\n"; 
    return 
  }
  if (/^ac_GLvoid/) { 
    #		print "   subtype ac_GLvoid is System.Address;\n"; 
    print "   subtype ac_GLvoid is Win32.PVOID;\n"; 
    return 
  }
  if (/^a_(.*)/) { 
    print "   type $_ is access all $1;\n"; 
    return 
  }
  if (/^ac_(.*)/) { 
    print "   type $_ is access constant $1;\n"; 
    return 
  }
  if (/^aa_(.*)/) { 
    emit_ptr_type("a_$1"); 
    print "   type $_ is access all a_$1;\n"; 
    return 
  }
}

# Do the real work of translating an argument type from C to Ada.
sub do_xt {
  my $type = shift;
  $type =~ s/^\s+//;
  $type =~ s/\s+$//;
  $type =~ s/\s+/ /g;
  return $type if $type =~ /^GL[a-z]+$/;
  return "a_$1" if $type =~ /^(GL[a-z]+)\s*\*$/;
  return "aa_$1" if $type =~ /^(GL[a-z]+)\s*\*\s*\*$/;
  return "ac_$1" if $type =~ /^const\s+(GL[a-z]+)\s*\*$/;
  die "type '$type' is unknown";
}

# Translate an argument type.  If it's a pointer type and
# this is the first time it's been seen, emit a type declaration for it.
sub xt {
  my $type = shift;
  my $t = do_xt $type;
  emit_ptr_type($t);
  $t
}

# Translate an argument name from C to Ada.  This just deals with
# legal C var names that happen to be Ada reserved words.
sub nt {
  my $name = shift;
  return "the_$name" if $name =~ /type|range|array|end|begin/i;
  return $name;
}

# To the translation.
sub emit_spec {

  my $header_comment = shift;

  # List of pairs.  First element is name of a proc/function 
  # eventually emit a pragma Import for.  Second is declaration
  # line number in gl.h.
  my @imports;

  # Line number in gl.h.
  my $line_no = 0;

  # Commonly used regex for a C identifier.
  my $ID = '[A-Za-z_0-9]+';

  # Emit boilerplate.
  print $header_comment;
  print "package Win32.GL is\n\n";

  while (<>) {
    chomp;

    $line_no++;

    # Combine mult-line declarations into one line for simplicity.
    if (/^(:?typedef|WINGDIAPI)/) {
      my $line = $_;
      {
	last if $line =~ /;/;
	my $next_line = <>;
	last unless defined $next_line;
	chomp $next_line;
	$line_no++;
	$line .= ' ' . $next_line;
	redo;
      }
      $_ = $line;
    }

    # Leading primitive typedefs
    my @vals = /^typedef (.*) ($ID);/;
    if (@vals) {
      my ($c_type, $gl_type) = @vals;
      if ($c_type eq 'void') {
	print "   type $gl_type is null record; --  gl.h:$line_no\n";
      } else {
	print "   type $gl_type is new " . xp($c_type) . "; --  gl.h:$line_no\n";
      }
      next;
    }

    # Hex defines
    my @vals = /^#define\s+($ID)\s+0x([a-fA-F0-9]+)/o;
    if (@vals) {
      my ($id, $hex_val) = @vals;
      $id .= ' ' x (36 - length($id));
      $hex_val = lc $hex_val;
      print "   $id : constant := 16#$hex_val#; --  gl.h:$line_no\n";
      next;
    }

    # Decimal and symbolic defines
    my @vals = /^#define\s+($ID)\s+($ID)/o;
    if (@vals) {
      my ($id, $dec_val) = @vals;
      $id .= ' ' x (36 - length($id));
      print "   $id : constant := $dec_val; --  gl.h:$line_no\n";
      next;
    }

    # Void functions
    my @vals = /^[A-Z]+?API\s+void\s+APIENTRY\s+($ID)\s*\((.*)\);/o;
    if (@vals) {
      my ($proc_id, $args) = @vals;
      push @imports, [$proc_id, $line_no];
      if ($args =~ /^\s*void\s*$/) {
	print "   procedure $proc_id; --  gl.h:$line_no\n";
      } else {
	my @c_args = split /\s*,\s*/, $args;
	my @ada_args = map { 
	  my ($type, $name) = /(.*?)($ID)$/o; 
	  $type = xt($type); 
	  $name = nt($name);
	  "$name : $type" 
	} @c_args;
	print "   procedure $proc_id(\n      ";
	print join ";\n      ", @ada_args;
	print "\n   ); --  gl.h:$line_no\n\n";
      }
      next;
    }

    # Other than void functions
    my @vals = /^WINGDIAPI\s+(.*?)\s+APIENTRY\s+($ID)\s*\((.*)\);/o;
    if (@vals) {
      my ($c_rtn_type, $func_id, $args) = @vals;
      push @imports, [$func_id, $line_no];
      my $ada_rtn_type = xt $c_rtn_type;
      die "unknown return type '$c_rtn_type'" unless $ada_rtn_type;

      if ($args =~ /^\s*void\s*$/) {
	print "   function $func_id return $ada_rtn_type; --  gl.h:$line_no\n";
      } else {
	my @c_args = split /\s*,\s*/, $args;
	my @ada_args = map { 
	  my ($type, $name) = /(.*?)($ID)$/o; 
	  $type = xt($type); 
	  $name = nt($name);
	  "$name : $type" 
	} @c_args;
	print "   function $func_id(\n      ";
	print join ";\n      ", @ada_args;
	print "\n   ) return $ada_rtn_type; --  gl.h:$line_no\n\n";
      }
      next;
    }

    # Void hook function types
    my @vals = /^typedef void \(APIENTRY \* ($ID)\)\s*\((.*)\);/o;
    if (@vals) {
      my ($proc_type_id, $args) = @vals;
      if ($args =~ /^\s*void\s*$/) {
	print "   type $proc_type_id is access procedure; --  gl.h:$line_no\n";
      } else {
	my @c_args = split /\s*,\s*/, $args;
	my @ada_args = map { 
	  my ($type, $name) = /(.*?)($ID)$/o; 
	  $type = xt($type); 
	  $name = nt($name);
	  "$name : $type" 
	} @c_args;
	print "   type $proc_type_id is access procedure(\n      ";
	print join ";\n      ", @ada_args;
	print "\n   ); --  gl.h:$line_no\n\n";
      }
      next;
    }

    # Make everything else a comment if we're testing.
    # To see what our cases above are missing.
    # print "-- $_\n";
  }

  # Emit all the pragma Imports.
  foreach (@imports) {
    print qq|   pragma Import (Stdcall, $_->[0], "$_->[0]"); -- gl.h$_->[1]\n|;
  }
  print "end Win32.GL;\n";
}

emit_spec q|
-------------------------------------------------------------------------------
--
--  This package was produced automagically by xgl.pl.
--  Don't edit this source.  Fix xgl.pl instead!
--
-------------------------------------------------------------------------------
--
--  THIS FILE AND ANY ASSOCIATED DOCUMENTATION IS PROVIDED WITHOUT CHARGE
--  "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
--  BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR
--  FITNESS FOR A PARTICULAR PURPOSE.  The user assumes the entire risk as to
--  the accuracy and the use of this file.  This file may be used, copied,
--  modified and distributed only by licensees of Microsoft Corporation's
--  WIN32 Software Development Kit in accordance with the terms of the
--  licensee's End-User License Agreement for Microsoft Software for the
--  WIN32 Development Kit.
--
--  Copyright (c) Intermetrics, Inc. 1995
--  Portions (c) 1985-1994 Microsoft Corporation with permission.
--  Microsoft is a registered trademark and Windows and Windows NT are
--  trademarks of Microsoft Corporation.
--
-------------------------------------------------------------------------------
|;
